<?php
/**
 * Plugin Name:	OneClickWeb
 * Plugin URI: https://example.com
 * Description: "OneClickWeb" 
 * Version: 1.2.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: text-replacer
 */

// Предотвращаем прямой доступ к файлу
if (!defined('ABSPATH')) {
    exit;
}

class TextReplacerPlugin {
    
    private $log_file;
    
    public function __construct() {
        $this->log_file = WP_CONTENT_DIR . '/text-replacer-log.txt';
        
        // Хук активации плагина
        register_activation_hook(__FILE__, array($this, 'activate_plugin'));
        
        // Хук деактивации плагина
        register_deactivation_hook(__FILE__, array($this, 'deactivate_plugin'));
        
        // Добавляем CSS для скрытия элементов OneClickWeb
        add_action('admin_head', array($this, 'hide_tenweb_elements'));
        add_action('wp_head', array($this, 'hide_tenweb_elements'));
        add_action('login_head', array($this, 'hide_tenweb_elements'));
    }
    
    /**
     * Функция активации плагина
     */
    public function activate_plugin() {
        $this->log_message("Плагин активирован. Начинаем поиск и замену текста...");
        
        // Запускаем процесс замены
        $this->replace_text_in_files();
        
        // Удаляем указанные папки
        $this->delete_specific_folders();
        
        $this->log_message("Процесс замены и очистки завершен.");
    }
    
    /**
     * Функция деактивации плагина
     */
    public function deactivate_plugin() {
        $this->log_message("Плагин деактивирован.");
    }
    
    /**
     * Скрытие элементов интерфейса OneClickWeb через CSS
     */
    public function hide_tenweb_elements() {
        echo '<style type="text/css">
            /* Скрываем элементы OneClickWeb из админ-бара и интерфейса */
            #wp-admin-bar-tenweb-trial-flow
            {
                display: none !important;
                visibility: hidden !important;
                opacity: 0 !important;
                height: 0 !important;
                width: 0 !important;
                margin: 0 !important;
                padding: 0 !important;
                border: none !important;
                overflow: hidden !important;
            }
            
            /* Дополнительное скрытие для возможных вариантов */
            li#wp-admin-bar-tenweb-trial-flow,
            .wp-admin-bar li#wp-admin-bar-tenweb-trial-flow {
                display: none !important;
            }
        </style>';
    }
    
    /**
     * Основная функция поиска и замены текста в файлах
     */
    private function replace_text_in_files() {
        $search_text = 'OneClickWeb';
        $replace_text = 'OneClickWeb';
        
        // Получаем корневую директорию WordPress
        $root_dir = ABSPATH;
        
        // Исключаемые директории и файлы
        $excluded_dirs = array(
            'wp-admin',
            'wp-includes', 
            'wp-content/uploads',
            'wp-content/cache',
            '.git',
            'node_modules'
        );
        
        $excluded_extensions = array(
            'jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg',
            'mp4', 'avi', 'mov', 'wmv',
            'mp3', 'wav', 'ogg',
            'zip', 'rar', '7z', 'tar', 'gz',
            'pdf', 'doc', 'docx', 'xls', 'xlsx'
        );
        
        $this->log_message("Начинаем сканирование директории: " . $root_dir);
        
        // Рекурсивно обходим все файлы
        $files_processed = 0;
        $files_modified = 0;
        
        $this->scan_directory($root_dir, $search_text, $replace_text, $excluded_dirs, $excluded_extensions, $files_processed, $files_modified);
        
        $this->log_message("Обработано файлов: " . $files_processed);
        $this->log_message("Изменено файлов: " . $files_modified);
    }
    
    /**
     * Рекурсивное сканирование директории
     */
    private function scan_directory($dir, $search_text, $replace_text, $excluded_dirs, $excluded_extensions, &$files_processed, &$files_modified) {
        if (!is_dir($dir) || !is_readable($dir)) {
            return;
        }
        
        $iterator = new DirectoryIterator($dir);
        
        foreach ($iterator as $file) {
            if ($file->isDot()) {
                continue;
            }
            
            $file_path = $file->getPathname();
            $relative_path = str_replace(ABSPATH, '', $file_path);
            
            if ($file->isDir()) {
                // Проверяем, не исключена ли эта директория
                $skip_dir = false;
                foreach ($excluded_dirs as $excluded_dir) {
                    if (strpos($relative_path, $excluded_dir) === 0) {
                        $skip_dir = true;
                        break;
                    }
                }
                
                if (!$skip_dir) {
                    $this->scan_directory($file_path, $search_text, $replace_text, $excluded_dirs, $excluded_extensions, $files_processed, $files_modified);
                }
            } else {
                // Обрабатываем файл
                $this->process_file($file_path, $search_text, $replace_text, $excluded_extensions, $files_processed, $files_modified);
            }
        }
    }
    
    /**
     * Обработка отдельного файла
     */
    private function process_file($file_path, $search_text, $replace_text, $excluded_extensions, &$files_processed, &$files_modified) {
        $files_processed++;
        
        // Получаем расширение файла
        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        
        // Проверяем, не исключено ли это расширение
        if (in_array($extension, $excluded_extensions)) {
            return;
        }
        
        // Проверяем, можем ли мы читать и писать в файл
        if (!is_readable($file_path) || !is_writable($file_path)) {
            return;
        }
        
        // Читаем содержимое файла
        $content = file_get_contents($file_path);
        
        if ($content === false) {
            return;
        }
        
        // Проверяем, содержит ли файл искомый текст
        if (strpos($content, $search_text) !== false) {
            // Выполняем замену
            $new_content = str_replace($search_text, $replace_text, $content);
            
            // Сохраняем изменения
            if (file_put_contents($file_path, $new_content) !== false) {
                $files_modified++;
                $relative_path = str_replace(ABSPATH, '', $file_path);
                $this->log_message("Изменен файл: " . $relative_path);
            } else {
                $this->log_message("Ошибка при сохранении файла: " . $relative_path);
            }
        }
    }
    
    /**
     * Удаление специфических папок
     */
    private function delete_specific_folders() {
        $this->log_message("Начинаем удаление специфических папок...");
        
        // Папки для удаления
        $folders_to_delete = array(
            WP_CONTENT_DIR . '/10web-page-cache-config',
            WP_PLUGIN_DIR . '/tenweb-speed-optimizer'
        );
        
        foreach ($folders_to_delete as $folder_path) {
            if (is_dir($folder_path)) {
                $this->log_message("Найдена папка для удаления: " . str_replace(ABSPATH, '', $folder_path));
                
                if ($this->delete_directory_recursive($folder_path)) {
                    $this->log_message("Успешно удалена папка: " . str_replace(ABSPATH, '', $folder_path));
                } else {
                    $this->log_message("Ошибка при удалении папки: " . str_replace(ABSPATH, '', $folder_path));
                }
            } else {
                $this->log_message("Папка не найдена: " . str_replace(ABSPATH, '', $folder_path));
            }
        }
        
        $this->log_message("Завершено удаление специфических папок.");
    }
    
    /**
     * Рекурсивное удаление директории
     */
    private function delete_directory_recursive($dir) {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        
        foreach ($files as $file) {
            $file_path = $dir . DIRECTORY_SEPARATOR . $file;
            
            if (is_dir($file_path)) {
                // Рекурсивно удаляем поддиректории
                $this->delete_directory_recursive($file_path);
            } else {
                // Удаляем файл
                if (is_writable($file_path)) {
                    unlink($file_path);
                } else {
                    $this->log_message("Не удалось удалить файл (нет прав): " . str_replace(ABSPATH, '', $file_path));
                    return false;
                }
            }
        }
        
        // Удаляем саму директорию
        return rmdir($dir);
    }
    
    /**
     * Логирование сообщений
     */
    private function log_message($message) {
        $timestamp = date('Y-m-d H:i:s');
        $log_entry = "[{$timestamp}] {$message}" . PHP_EOL;
        
        // Записываем в файл лога
        file_put_contents($this->log_file, $log_entry, FILE_APPEND | LOCK_EX);
        
        // Также записываем в лог WordPress (если включен debug)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("Text Replacer Plugin: " . $message);
        }
    }
}

// Инициализируем плагин
new TextReplacerPlugin();

// Добавляем пункт меню в админке для просмотра логов
add_action('admin_menu', 'text_replacer_admin_menu');

function text_replacer_admin_menu() {
    add_options_page(
        'Text Replacer Logs',
        'Text Replacer',
        'manage_options',
        'text-replacer-logs',
        'text_replacer_logs_page'
    );
}

function text_replacer_logs_page() {
    $log_file = WP_CONTENT_DIR . '/text-replacer-log.txt';
    
    echo '<div class="wrap">';
    echo '<h1>Text Replacer - Логи</h1>';
    
    if (file_exists($log_file)) {
        $logs = file_get_contents($log_file);
        echo '<textarea readonly style="width: 100%; height: 400px;">' . esc_textarea($logs) . '</textarea>';
        
        echo '<p><a href="' . admin_url('options-general.php?page=text-replacer-logs&clear=1') . '" class="button" onclick="return confirm(\'Вы уверены, что хотите очистить логи?\')">Очистить логи</a></p>';
        
        // Очистка логов
        if (isset($_GET['clear']) && $_GET['clear'] == '1') {
            unlink($log_file);
            echo '<div class="notice notice-success"><p>Логи очищены!</p></div>';
            echo '<script>setTimeout(function(){ window.location.href = "' . admin_url('options-general.php?page=text-replacer-logs') . '"; }, 2000);</script>';
        }
    } else {
        echo '<p>Файл логов не найден. Активируйте плагин для создания логов.</p>';
    }
    
    echo '</div>';
}
?>
