#!/bin/bash


## part 1 - tailwind css build

set -e

# Colors for output
RED="\033[0;31m"
GREEN="\033[0;32m"
YELLOW="\033[1;33m"
BLUE="\033[0;34m"
NC="\033[0m" # No Color

echo -e "${BLUE}🪝 WVC Theme pre-commit hook${NC}"

THEME_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/" && pwd)"
echo "Debug: Theme directory: $THEME_DIR"


# Check if any relevant files have been modified
# Debug: Print current directory and git status

# Use git diff-index instead of git diff --staged for better compatibility
RELEVANT_FILES=$(git diff-index --cached --name-only HEAD | grep -E "(assets/css/|\.css$|tailwind\.config\.js|postcss\.config\.js|package\.json)" || true)

if [ -n "$RELEVANT_FILES" ]; then
    echo -e "${YELLOW}📝 Detected changes to CSS or config files:${NC}"
    echo "$RELEVANT_FILES" | sed "s/^/     /"
    echo ""
    
    echo -e "${YELLOW}🔨 Running CSS build...${NC}"
    
    # Run the build script
    if [ -f "build/build-css.sh" ]; then
        # Make sure it's executable
        chmod +x build/build-css.sh
        
        # Run the build
        if ./build/build-css.sh; then
            echo -e "${GREEN}✅ CSS build successful${NC}"
            
            # Add the generated files to the commit if they exist
          
            if [ -f "style.css" ]; then
                git add style.css
                echo -e "${GREEN}📄 Added style.css to commit${NC}"
            fi
        else
            echo -e "${RED}❌ CSS build failed${NC}"
            exit 1
        fi
    else
        echo -e "${RED}❌ Error: build/build-css.sh not found${NC}"
        exit 1
    fi
else
    echo -e "${BLUE}📝 No CSS or config files changed, skipping CSS build${NC}"
fi

echo -e "${GREEN}✅ Pre-commit hook - part 1 - node scripts - completed${NC}"


# part 2 - php checks






# Exit on any error
# Removing set -e to allow checking all files
# set -e


# Initialize error counter
ERROR_COUNT=0

echo -e "${BLUE}🔍 Checking php files...${NC}"
# Store list of PHP files that are staged for commit
STAGED_PHP_FILES=$(git diff-index --cached --name-only --diff-filter=d HEAD | grep ".php$" || true)

# Check if composer.json is modified
COMPOSER_MODIFIED=$(git diff-index --cached --name-only --diff-filter=d HEAD | grep "composer.json$" || true)
# Function to check if a command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

echo -e "${BLUE}🔍 Checking composer.json...${NC}"

# Check if composer is installed
if ! command_exists composer; then
    echo -e "${RED}❌ Error: Composer is not installed. Please install Composer first.${NC}"
    echo -e "${RED}   Visit https://getcomposer.org/download/ for installation instructions.${NC}"
    exit 1
fi


# Check composer.json syntax if modified
if [ ! -z "$COMPOSER_MODIFIED" ]; then
    echo -e "${BLUE}🔍 Checking composer.json syntax...${NC}"
    composer validate --no-check-all --no-check-publish
    if [ $? -ne 0 ]; then
        echo -e "${RED}❌ composer.json validation failed!${NC}"
        ERROR_COUNT=$((ERROR_COUNT + 1))
    fi
fi

# Check if composer dependencies are installed
if [ ! -d "vendor" ]; then
    echo -e "${YELLOW}📦 Installing Composer dependencies...${NC}"
    composer install --no-interaction
fi

# Check for outdated dependencies
echo -e "${BLUE}🔍 Checking for outdated dependencies...${NC}"
composer outdated --direct --no-interaction || true

# If no PHP files are staged, exit successfully
if [ -z "$STAGED_PHP_FILES" ]; then
    echo -e "${BLUE}📝 No PHP files staged for commit. Skipping PHP checks.${NC}"
    exit 0
fi

echo -e "${BLUE}🚀 Running pre-commit checks on PHP files...${NC}"

# PHP Lint
echo -e "${BLUE}🔍 Running PHP Lint...${NC}"
for FILE in $STAGED_PHP_FILES; do
    if [ -f "$FILE" ]; then
        echo -e "${YELLOW}Checking: $FILE${NC}"
        php -l "$FILE"
        if [ $? -ne 0 ]; then
            echo -e "${RED}❌ PHP Lint failed for: $FILE${NC}"
            ERROR_COUNT=$((ERROR_COUNT + 1))
        else
            echo -e "${GREEN}✅ PHP Lint passed for: $FILE${NC}"
        fi
    fi
done
echo -e "${GREEN}✅ PHP Lint completed!${NC}"


echo -e "${GREEN}✅ Pre-commit hook - part 2 - php checks - completed${NC}"


exit 0
# TODO - shortcut as the code below is not working

# Run PHPCBF on each staged file
for FILE in $STAGED_PHP_FILES; do
    if [ -f "$FILE" ]; then
        echo -e "${YELLOW}🔧 Processing $FILE with PHPCBF${NC}"
        ./vendor/bin/phpcbf -v "$FILE" 2>&1 || true
        # Re-add the file to staging after PHPCBF modifications
        git add "$FILE"
    fi
done

exit 1

# PHP CodeSniffer
if command_exists ./vendor/bin/phpcs; then
    echo -e "${BLUE}🔍 Running PHP CodeSniffer...${NC}"
    HAS_ERRORS=0
    for FILE in $STAGED_PHP_FILES; do
        if [ -f "$FILE" ]; then
            if ! ./vendor/bin/phpcs --standard=phpcs.xml.dist "$FILE"; then
                echo -e "${RED}❌ PHPCS check failed for: $FILE${NC}"
                echo -e "${RED}   Please fix the above errors and try committing again.${NC}"
                ERROR_COUNT=$((ERROR_COUNT + 1))
                HAS_ERRORS=1
            fi
        fi
    done
    if [ $HAS_ERRORS -eq 1 ]; then
        echo -e "${RED}❌ PHPCS found errors in one or more files.${NC}"
    fi
    echo -e "${GREEN}✅ PHPCS completed!${NC}"
else
    echo -e "${YELLOW}📦 Installing PHP_CodeSniffer and dependencies...${NC}"
    composer require --dev squizlabs/php_codesniffer
    composer require --dev wp-coding-standards/wpcs
    composer require --dev phpcsstandards/phpcsutils
    composer require --dev phpcsstandards/phpcsextra
    composer require --dev dealerdirect/phpcodesniffer-composer-installer
fi

# PHP Mess Detector
if command_exists ./vendor/bin/phpmd; then
    echo -e "${BLUE}🔍 Running PHP Mess Detector...${NC}"
    for FILE in $STAGED_PHP_FILES; do
        if [ -f "$FILE" ]; then
            ./vendor/bin/phpmd "$FILE" text cleancode,codesize,controversial,design,naming,unusedcode || true
            # Note: PHPMD errors are treated as warnings, not errors
        fi
    done
    echo -e "${GREEN}✅ PHPMD completed!${NC}"
else
    echo -e "${YELLOW}📦 Installing PHP Mess Detector...${NC}"
    composer require --dev phpmd/phpmd
fi

exit 1
# PHP Stan
if command_exists ./vendor/bin/phpstan; then
    echo -e "${BLUE}🔍 Running PHPStan...${NC}"
    HAS_ERRORS=0
    
    # Create PHPStan config if it doesn't exist
    if [ ! -f "phpstan.neon" ]; then
        echo -e "${YELLOW}📝 Creating PHPStan configuration file...${NC}"
        cat > phpstan.neon << 'EOL'
parameters:
    level: 5
    paths:
        - .
    bootstrapFiles:
        - vendor/php-stubs/wordpress-stubs/wordpress-stubs.php
    excludePaths:
        - vendor/*
    ignoreErrors:
        - '#Function [a-zA-Z0-9\\_]+ not found\.#'
        - '#Function apply_filters(_ref_array)? invoked with [0-9]+ parameters?, 2 required\.#'
        - '#Function do_action(_ref_array)? invoked with [0-9]+ parameters?, 1 required\.#'
        - '#Constant [A-Z0-9\\_]+ not found\.#'
EOL
        # Install WordPress stubs for PHPStan
        composer require --dev php-stubs/wordpress-stubs
        composer require --dev szepeviktor/phpstan-wordpress
    fi

    for FILE in $STAGED_PHP_FILES; do
        if [ -f "$FILE" ]; then
            if ! ./vendor/bin/phpstan analyse --configuration=phpstan.neon "$FILE"; then
                echo -e "${RED}❌ PHPStan check failed for: $FILE${NC}"
                echo -e "${RED}   Please fix the above errors and try committing again.${NC}"
                ERROR_COUNT=$((ERROR_COUNT + 1))
                HAS_ERRORS=1
            fi
        fi
    done
    if [ $HAS_ERRORS -eq 1 ]; then
        echo -e "${RED}❌ PHPStan found errors in one or more files.${NC}"
    fi
    echo -e "${GREEN}✅ PHPStan completed!${NC}"
else
    echo -e "${YELLOW}📦 Installing PHPStan and WordPress support...${NC}"
    composer require --dev phpstan/phpstan
    composer require --dev php-stubs/wordpress-stubs
    composer require --dev szepeviktor/phpstan-wordpress
fi

if [ $ERROR_COUNT -gt 0 ]; then
    echo -e "${RED}❌ Found $ERROR_COUNT error(s). Please fix them and try committing again.${NC}"
    exit 1
else
    echo -e "${GREEN}✅ All pre-commit checks passed!${NC}"
    exit 0
fi