<?php
/**
 * Bulk Operations REST API Controller
 *
 * This class provides REST API endpoints for bulk operations including
 * template imports, page creation, and batch processing functionality.
 * It handles authentication, validation, and batch operations management.
 *
 * @package    WVC_Theme
 * @subpackage Bulk_Operations
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

class WVC_Bulk_Operations_REST_API
{
    
    private $namespace = "wvc/v1";
    private $base = "structure";
    
    public function __construct()
    {
        add_action("rest_api_init", array($this, "register_routes"));
        
        // Initialize template title uniqueness hooks
        WVC_Structure_Validator::init_wordpress_hooks();
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Structure endpoints
        register_rest_route(
            $this->namespace, "/" . $this->base, array(
            array(
                "methods" => WP_REST_Server::READABLE,
                "callback" => array($this, "get_structure"),
                "permission_callback" => array($this, "get_permissions_check")
            ),
            array(
                "methods" => WP_REST_Server::CREATABLE,
                "callback" => array($this, "update_structure"),
                "permission_callback" => array($this, "create_permissions_check"),
                "args" => array(
                    "pages" => array(
                        "type" => "array",
                        "description" => __("Array of page data", "wvc-theme")
                    ),
                    "templates" => array(
                        "type" => "array",
                        "description" => __("Array of template data", "wvc-theme")
                    )
                )
            )
            )
        );
    }
    
    /**
     * Get complete site structure
     */
    public function get_structure($request)
    {
        $structure = array(
            "pages" => array(),
            "templates" => array()
        );
        
        // Get all pages
        $pages = get_posts(array(
            "post_type" => "page",
            "post_status" => array("publish", "draft", "private"),
            "posts_per_page" => -1,
            "orderby" => "menu_order",
            "order" => "ASC"
        ));
        
        foreach ($pages as $page) {
            $structure["pages"][] = $this->prepare_page_response($page);
        }
        
        // Get all templates
        $templates = get_posts(array(
            "post_type" => "wvc_template",
            "post_status" => array("publish", "draft", "private"),
            "posts_per_page" => -1,
            "orderby" => "menu_order",
            "order" => "ASC"
        ));
        
        foreach ($templates as $template) {
            $structure["templates"][] = $this->prepare_template_response($template);
        }
        
        return rest_ensure_response($structure);
    }
    
    /**
     * Get single template
     */
    public function get_template($request)
    {
        $id = $request->get_param("id");
        $post = get_post($id);
        
        if (!$post || $post->post_type !== "wvc_template") {
            return new WP_Error(
                "wvc_template_not_found",
                __("Template not found", "wvc-theme"),
                array("status" => 404)
            );
        }
        
        return rest_ensure_response($this->prepare_template_response($post));
    }
    
    /**
     * Create new template
     */
    public function create_template($request)
    {
        $title = $request->get_param("title");
        $template_id = $request->get_param("template_id");
        $template_data = $request->get_param("template_data");
        $file_path = $request->get_param("file_path");
        
        // Validate template_id
        if (!$this->is_valid_template_id($template_id)) {
            return new WP_Error(
                "wvc_invalid_template_id",
                __("Invalid template type", "wvc-theme"),
                array("status" => 400)
            );
        }
        
        // Check if template of this type already exists
        if ($this->template_type_exists($template_id)) {
            return new WP_Error(
                "wvc_template_exists",
                __("Template of this type already exists", "wvc-theme"),
                array("status" => 409)
            );
        }
        
        $post_data = array(
            "post_type" => "wvc_template",
            "post_title" => $title,
            "post_status" => "publish"
        );
        
        $post_id = wp_insert_post($post_data);
        
        if (is_wp_error($post_id)) {
            return $post_id;
        }
        
        // Save meta fields
        update_post_meta($post_id, "_wvc_template_id", $template_id);
        
        if ($template_data) {
            update_post_meta($post_id, "_wvc_template", wp_json_encode($template_data));
        }
        
        if ($file_path) {
            update_post_meta($post_id, "_wvc_template_file_path", $file_path);
        }
        
        // Set hierarchy relationships
        $this->set_template_hierarchy($post_id, $template_id);
        
        $post = get_post($post_id);
        return rest_ensure_response($this->prepare_template_response($post));
    }
    
    /**
     * Update template
     */
    public function update_template($request)
    {
        $id = $request->get_param("id");
        $post = get_post($id);
        
        if (!$post || $post->post_type !== "wvc_template") {
            return new WP_Error(
                "wvc_template_not_found",
                __("Template not found", "wvc-theme"),
                array("status" => 404)
            );
        }
        
        $update_data = array("ID" => $id);
        
        // Update title if provided
        $title = $request->get_param("title");
        if ($title !== null) {
            $update_data["post_title"] = $title;
        }
        
        // Update post if needed
        if (count($update_data) > 1) {
            $result = wp_update_post($update_data);
            if (is_wp_error($result)) {
                return $result;
            }
        }
        
        // Update template_id if provided
        $template_id = $request->get_param("template_id");
        if ($template_id !== null) {
            if (!$this->is_valid_template_id($template_id)) {
                return new WP_Error(
                    "wvc_invalid_template_id",
                    __("Invalid template type", "wvc-theme"),
                    array("status" => 400)
                );
            }
            
            $existing_template_id = get_post_meta($id, "_wvc_template_id", true);
            if ($template_id !== $existing_template_id && $this->template_type_exists($template_id)) {
                return new WP_Error(
                    "wvc_template_exists",
                    __("Template of this type already exists", "wvc-theme"),
                    array("status" => 409)
                );
            }
            
            update_post_meta($id, "_wvc_template_id", $template_id);
            $this->set_template_hierarchy($id, $template_id);
        }
        
        // Update template data if provided
        $template_data = $request->get_param("template_data");
        if ($template_data !== null) {
            update_post_meta($id, "_wvc_template", wp_json_encode($template_data));
        }
        
        // Update file path if provided
        $file_path = $request->get_param("file_path");
        if ($file_path !== null) {
            update_post_meta($id, "_wvc_template_file_path", $file_path);
        }
        
        $updated_post = get_post($id);
        return rest_ensure_response($this->prepare_template_response($updated_post));
    }
    
    /**
     * Delete template
     */
    public function delete_template($request)
    {
        $id = $request->get_param("id");
        $force = $request->get_param("force");
        
        $post = get_post($id);
        
        if (!$post || $post->post_type !== "wvc_template") {
            return new WP_Error(
                "wvc_template_not_found",
                __("Template not found", "wvc-theme"),
                array("status" => 404)
            );
        }
        
        $result = wp_delete_post($id, $force);
        
        if (!$result) {
            return new WP_Error(
                "wvc_template_delete_failed",
                __("Failed to delete template", "wvc-theme"),
                array("status" => 500)
            );
        }
        
        return rest_ensure_response(array(
            "deleted" => true,
            "previous" => $this->prepare_template_response($post)
        ));
    }
    
    /**
     * Get template hierarchy tree
     */
    public function get_hierarchy($request)
    {
        $templates = get_posts(array(
            "post_type" => "wvc_template",
            "post_status" => "any",
            "posts_per_page" => -1,
            "orderby" => "menu_order",
            "order" => "ASC"
        ));
        
        $hierarchy = array();
        $template_map = array();
        
        // First pass: create all templates
        foreach ($templates as $post) {
            $template_data = $this->prepare_template_response($post);
            $template_data["children"] = array();
            $template_map[$post->ID] = $template_data;
        }
        
        // Second pass: build hierarchy
        foreach ($templates as $post) {
            if ($post->post_parent == 0) {
                $hierarchy[] = &$template_map[$post->ID];
            } else {
                if (isset($template_map[$post->post_parent])) {
                    $template_map[$post->post_parent]["children"][] = &$template_map[$post->ID];
                }
            }
        }
        
        return rest_ensure_response($hierarchy);
    }
    
    /**
     * Save complete site structure
     */
    public function update_structure($request)
    {
        $pages = $request->get_param("pages") ?: array();
        $templates = $request->get_param("templates") ?: array();
        
        $result = array(
            "pages" => array(
                "created" => array(),
                "updated" => array(),
                "errors" => array()
            ),
            "templates" => array(
                "created" => array(),
                "updated" => array(),
                "deleted" => array(),
                "errors" => array()
            )
        );
        
        // Process pages (standard ID-based operations)
        if (!empty($pages)) {
            $result["pages"] = $this->process_pages_bulk($pages);
        }
        
        // Process templates (title-based operations with cleanup)
        if (!empty($templates)) {
            $result["templates"] = $this->process_templates_bulk($templates);
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Process pages bulk operations
     */
    private function process_pages_bulk($pages)
    {
        $result = array(
            "created" => array(),
            "updated" => array(),
            "errors" => array()
        );
        
        foreach ($pages as $page_data) {
            try {
                $page_id = isset($page_data["id"]) ? intval($page_data["id"]) : 0;
                
                $post_data = array(
                    "post_type" => "page",
                    "post_title" => $page_data["title"] ?? "",
                    "post_content" => $page_data["content-managers"] ?? "",
                    "post_excerpt" => $page_data["excerpt"] ?? "",
                    "post_status" => $page_data["status"] ?? "publish",
                    "menu_order" => $page_data["menu_order"] ?? 0
                );
                
                if (!empty($page_data["slug"])) {
                    $post_data["post_name"] = $page_data["slug"];
                }
                
                if ($page_id > 0) {
                    // Update existing page
                    $post_data["ID"] = $page_id;
                    $updated_id = wp_update_post($post_data);
                    
                    if (is_wp_error($updated_id)) {
                        $result["errors"][] = array(
                            "id" => $page_id,
                            "title" => $page_data["title"] ?? "",
                            "error" => $updated_id->get_error_message()
                        );
                    } else {
                        $result["updated"][] = array(
                            "id" => $page_id,
                            "title" => $post_data["post_title"]
                        );
                    }
                } else {
                    // Create new page
                    $new_id = wp_insert_post($post_data);
                    
                    if (is_wp_error($new_id)) {
                        $result["errors"][] = array(
                            "title" => $page_data["title"] ?? "",
                            "error" => $new_id->get_error_message()
                        );
                    } else {
                        $result["created"][] = array(
                            "id" => $new_id,
                            "title" => $post_data["post_title"]
                        );
                    }
                }
                
                
            } catch (Exception $e) {
                $result["errors"][] = array(
                    "id" => $page_id,
                    "title" => $page_data["title"] ?? "",
                    "error" => $e->getMessage()
                );
            }
        }
        
        // Remove pages not in input
        $input_page_ids = $this->extract_page_ids_from_input($pages, $result);
        $this->cleanup_orphaned_pages($input_page_ids, $result);
        
        return $result;
    }
    
    /**
     * Process templates bulk operations with title-based logic
     */
    private function process_templates_bulk($templates)
    {
        $result = array(
            "created" => array(),
            "updated" => array(),
            "deleted" => array(),
            "errors" => array()
        );
        
        // Validate template titles within input array only (not against database since we handle updates)
        $input_template_titles = array();
        $seen_titles = array();
        $valid_templates = array();
        
        foreach ($templates as $index => $template_data) {
            $title = $template_data["title"] ?? $template_data["name"] ?? "";
            
            // Check for empty title
            if (empty($title)) {
                $result["errors"][] = array(
                    "index" => $index,
                    "title" => "",
                    "error" => __("Template missing title", "wvc-theme")
                );
                continue;
            }
            
            // Check for duplicate titles within input array
            if (in_array($title, $seen_titles)) {
                $result["errors"][] = array(
                    "index" => $index,
                    "title" => $title,
                    "error" => __("Duplicate template title in input", "wvc-theme")
                );
                continue;
            }
            
            $seen_titles[] = $title;
            $input_template_titles[] = $title;
            $valid_templates[] = $template_data;
        }
        
        // Process valid templates
        foreach ($valid_templates as $template_data) {
            try {
                $title = $template_data["title"] ?? $template_data["name"] ?? "";
                
                if (empty($title)) {
                    $result["errors"][] = array(
                        "title" => "",
                        "error" => __("Template missing title", "wvc-theme")
                    );
                    continue;
                }
                
                // Check if template exists by title
                $existing_template = WVC_Structure_Validator::get_template_by_title($title);
                
                $post_data = array(
                    "post_type" => "wvc_template",
                    "post_title" => $title,
                    "post_content" => $template_data["content-managers"] ?? "",
                    "post_excerpt" => $template_data["description"] ?? "",
                    "post_status" => $template_data["status"] ?? "publish"
                );
                
                if ($existing_template) {
                    // Update existing template
                    $post_data["ID"] = $existing_template->ID;
                    $updated_id = wp_update_post($post_data);
                    
                    if (is_wp_error($updated_id)) {
                        $result["errors"][] = array(
                            "title" => $title,
                            "error" => $updated_id->get_error_message()
                        );
                    } else {
                        $result["updated"][] = array(
                            "id" => $existing_template->ID,
                            "title" => $title
                        );
                        
                        // Update meta fields
                        $this->update_template_meta($existing_template->ID, $template_data);
                    }
                } else {
                    // Create new template
                    $new_id = wp_insert_post($post_data);
                    
                    if (is_wp_error($new_id)) {
                        $result["errors"][] = array(
                            "title" => $title,
                            "error" => $new_id->get_error_message()
                        );
                    } else {
                        $result["created"][] = array(
                            "id" => $new_id,
                            "title" => $title
                        );
                        
                        // Set meta fields
                        $this->update_template_meta($new_id, $template_data);
                    }
                }
                
            } catch (Exception $e) {
                $result["errors"][] = array(
                    "title" => $template_data["title"] ?? $template_data["name"] ?? "",
                    "error" => $e->getMessage()
                );
            }
        }
        
        // Remove templates not in input
        $this->cleanup_orphaned_templates($input_template_titles, $result);
        
        return $result;
    }
    
    /**
     * Update template meta fields
     */
    private function update_template_meta($post_id, $template_data)
    {
        // no metadata to update yet
        
    }
    
    /**
     * Remove templates that are not in the input list
     */
    private function cleanup_orphaned_templates($input_template_titles, &$result)
    {
        // Get all existing templates
        $existing_templates = get_posts(array(
            "post_type" => "wvc_template",
            "post_status" => array("publish", "draft", "private"),
            "posts_per_page" => -1,
            "fields" => "ids"
        ));
        
        foreach ($existing_templates as $template_id) {
            $template_post = get_post($template_id);
            if ($template_post && !in_array($template_post->post_title, $input_template_titles)) {
                // Template not in input - delete it
                $deleted = wp_delete_post($template_id, true); // Force delete
                
                if ($deleted) {
                    $result["deleted"][] = array(
                        "id" => $template_id,
                        "title" => $template_post->post_title
                    );
                } else {
                    $result["errors"][] = array(
                        "id" => $template_id,
                        "title" => $template_post->post_title,
                        "error" => __("Failed to delete orphaned template", "wvc-theme")
                    );
                }
            }
        }
    }
    
    /**
     * Extract page IDs from input data and results to determine which pages should be kept
     */
    private function extract_page_ids_from_input($pages, $result)
    {
        $page_ids = array();
        
        // Get IDs from input pages (existing pages being updated)
        foreach ($pages as $page_data) {
            $page_id = isset($page_data["id"]) ? intval($page_data["id"]) : 0;
            if ($page_id > 0) {
                $page_ids[] = $page_id;
            }
        }
        
        // Get IDs from newly created pages
        if (isset($result["created"])) {
            foreach ($result["created"] as $created_page) {
                if (isset($created_page["id"])) {
                    $page_ids[] = $created_page["id"];
                }
            }
        }
        
        return array_unique($page_ids);
    }
    
    /**
     * Remove pages that are not in the input list
     */
    private function cleanup_orphaned_pages($input_page_ids, &$result)
    {
        // Get all existing pages
        $existing_pages = get_posts(array(
            "post_type" => "page",
            "post_status" => array("publish", "draft", "private"),
            "posts_per_page" => -1,
            "fields" => "ids"
        ));
        
        foreach ($existing_pages as $page_id) {
            if (!in_array($page_id, $input_page_ids)) {
                // Page not in input - delete it
                $page_post = get_post($page_id);
                $deleted = wp_delete_post($page_id, true); // Force delete
                
                if ($deleted && $page_post) {
                    if (!isset($result["deleted"])) {
                        $result["deleted"] = array();
                    }
                    $result["deleted"][] = array(
                        "id" => $page_id,
                        "title" => $page_post->post_title
                    );
                } else {
                    $result["errors"][] = array(
                        "id" => $page_id,
                        "title" => $page_post ? $page_post->post_title : "",
                        "error" => __("Failed to delete orphaned page", "wvc-theme")
                    );
                }
            }
        }
    }
    
    /**
     * Bulk create templates (Legacy method)
     */
    public function bulk_create_templates($request)
    {
        $templates = $request->get_param("templates");
        $results = array();
        
        foreach ($templates as $template_data) {
            $template_request = new WP_REST_Request("POST", "/" . $this->namespace . "/" . $this->base);
            $template_request->set_body_params($template_data);
            
            $result = $this->create_template($template_request);
            
            if (is_wp_error($result)) {
                $results[] = array(
                    "error" => $result->get_error_message(),
                    "data" => $template_data
                );
            } else {
                $results[] = array(
                    "success" => true,
                    "data" => $result->get_data()
                );
            }
        }
        
        return rest_ensure_response($results);
    }
    
    /**
     * Get available template types
     */
    public function get_template_types($request)
    {
        // Get template hierarchy class to access template types
        $template_hierarchy = new WVC_Template_Post_Type();
        $types = $template_hierarchy->get_template_types();
        
        return rest_ensure_response($types);
    }
    
    /**
     * Helper methods
     */
    
    /**
     * Prepare template response data
     */
    private function prepare_template_response($post)
    {
        
        return array(
            "id" => $post->ID,
            "title" => $post->post_title,
            "content-managers" => $post->post_content,
            "status" => $post->post_status,
            "date_created" => $post->post_date,
            "date_modified" => $post->post_modified
        );
    }
    
    /**
     * Prepare page response data
     */
    private function prepare_page_response($post)
    {
        $page_template = get_page_template_slug($post->ID);
        
        return array(
            "id" => $post->ID,
            "title" => $post->post_title,
            "content-managers" => $post->post_content,
            "excerpt" => $post->post_excerpt,
            "slug" => $post->post_name,
            "status" => $post->post_status,
            "parent_id" => $post->post_parent,
            "menu_order" => $post->menu_order,
            "date_created" => $post->post_date,
            "date_modified" => $post->post_modified
        );
    }
    
    /**
     * Check if template ID is valid
     */
    private function is_valid_template_id($template_id)
    {
        $template_hierarchy = new WVC_Template_Post_Type();
        $types = $template_hierarchy->get_template_types();
        return array_key_exists($template_id, $types);
    }
    
    /**
     * Check if template type already exists
     */
    private function template_type_exists($template_id)
    {
        $posts = get_posts(array(
            "post_type" => "wvc_template",
            "meta_query" => array(
                array(
                    "key" => "_wvc_template_id",
                    "value" => $template_id,
                    "compare" => "="
                )
            ),
            "posts_per_page" => 1,
            "post_status" => "any"
        ));
        
        return !empty($posts);
    }
    
    /**
     * Set template hierarchy relationships
     */
    private function set_template_hierarchy($post_id, $template_id)
    {
        $template_hierarchy = new WVC_Template_Post_Type();
        $types = $template_hierarchy->get_template_types();
        
        if (!isset($types[$template_id])) {
            return;
        }
        
        $template_info = $types[$template_id];
        $parent_template_type = $template_info["parent"];
        
        // Calculate hierarchy level
        $level = $this->calculate_hierarchy_level($template_id, $types);
        update_post_meta($post_id, "_wvc_hierarchy_level", $level);
        
        // Find and set parent post if parent template type exists
        if ($parent_template_type) {
            $parent_posts = get_posts(array(
                "post_type" => "wvc_template",
                "meta_query" => array(
                    array(
                        "key" => "_wvc_template_id",
                        "value" => $parent_template_type,
                        "compare" => "="
                    )
                ),
                "posts_per_page" => 1,
                "post_status" => "any"
            ));
            
            if (!empty($parent_posts)) {
                wp_update_post(array(
                    "ID" => $post_id,
                    "post_parent" => $parent_posts[0]->ID
                ));
            }
        } else {
            // No parent - this is a root template
            wp_update_post(array(
                "ID" => $post_id,
                "post_parent" => 0
            ));
        }
    }
    
    /**
     * Calculate hierarchy level for a template type
     */
    private function calculate_hierarchy_level($template_id, $types, $visited = array())
    {
        if (in_array($template_id, $visited) || !isset($types[$template_id])) {
            return 0; // Prevent infinite loops or invalid templates
        }
        
        $parent = $types[$template_id]["parent"];
        if (!$parent) {
            return 0; // Root level
        }
        
        $visited[] = $template_id;
        return 1 + $this->calculate_hierarchy_level($parent, $types, $visited);
    }
    
    /**
     * Legacy methods (keeping for backward compatibility)
     */
    
    /**
     * Get pages
     */
    public function get_pages($request)
    {
        $pages = get_pages(array(
            "sort_column" => "menu_order",
            "sort_order" => "ASC"
        ));
        
        $structured_pages = array();
        foreach ($pages as $page) {
            $structured_pages[] = array(
                "id" => $page->ID,
                "title" => $page->post_title,
                "slug" => $page->post_name,
                "content-managers" => $page->post_content,
                "parent_page" => $page->post_parent ? get_post($page->post_parent)->post_name : "",
                "status" => $page->post_status,
                "menu_order" => $page->menu_order,
                "template" => get_page_template_slug($page->ID)
            );
        }
        
        return rest_ensure_response($structured_pages);
    }
    
    /**
     * Create pages
     */
    public function create_pages($request)
    {
        $pages = $request->get_param("pages");
        $created = array();
        $page_map = array();
        
        // Sort pages by hierarchy (parents first)
        usort($pages, function ($a, $b) {
            $a_has_parent = !empty($a["parent_page"]);
            $b_has_parent = !empty($b["parent_page"]);
            
            if ($a_has_parent && !$b_has_parent) {
                return 1;
            }
            if (!$a_has_parent && $b_has_parent) {
                return -1;
            }
            return 0;
        });
        
        foreach ($pages as $page_data) {
            $post_data = array(
                "post_type" => "page",
                "post_title" => $page_data["title"],
                "post_name" => $page_data["slug"],
                "post_content" => isset($page_data["content-managers"]) ? $page_data["content-managers"] : "",
                "post_status" => isset($page_data["status"]) ? $page_data["status"] : "publish",
                "menu_order" => isset($page_data["menu_order"]) ? $page_data["menu_order"] : 0
            );
            
            // Set parent if exists
            if (!empty($page_data["parent_page"]) && isset($page_map[$page_data["parent_page"]])) {
                $post_data["post_parent"] = $page_map[$page_data["parent_page"]];
            }
            
            $page_id = wp_insert_post($post_data);
            
            if (!is_wp_error($page_id)) {
                $page_map[$page_data["slug"]] = $page_id;
                
                
                $created[] = array(
                    "slug" => $page_data["slug"],
                    "id" => $page_id,
                    "success" => true
                );
            } else {
                $created[] = array(
                    "slug" => $page_data["slug"],
                    "error" => $page_id->get_error_message(),
                    "success" => false
                );
            }
        }
        
        return rest_ensure_response($created);
    }
    
    /**
     * Get posts
     */
    public function get_posts($request)
    {
        $posts = get_posts(array(
            "numberposts" => 50,
            "post_status" => "publish"
        ));
        
        $structured_posts = array();
        foreach ($posts as $post) {
            $categories = wp_get_post_categories($post->ID, array("fields" => "names"));
            $tags = wp_get_post_tags($post->ID, array("fields" => "names"));
            
            $structured_posts[] = array(
                "id" => $post->ID,
                "title" => $post->post_title,
                "content-managers" => $post->post_content,
                "excerpt" => $post->post_excerpt,
                "slug" => $post->post_name,
                "status" => $post->post_status,
                "categories" => $categories,
                "tags" => $tags,
                "featured_image" => get_the_post_thumbnail_url($post->ID, "full") ?: ""
            );
        }
        
        return rest_ensure_response($structured_posts);
    }
    
    /**
     * Create posts
     */
    public function create_posts($request)
    {
        $posts = $request->get_param("posts");
        $created = array();
        
        foreach ($posts as $post_data) {
            $post_array = array(
                "post_type" => "post",
                "post_title" => $post_data["title"],
                "post_content" => $post_data["content-managers"],
                "post_excerpt" => isset($post_data["excerpt"]) ? $post_data["excerpt"] : "",
                "post_status" => isset($post_data["status"]) ? $post_data["status"] : "publish"
            );
            
            if (!empty($post_data["slug"])) {
                $post_array["post_name"] = $post_data["slug"];
            }
            
            $post_id = wp_insert_post($post_array);
            
            if (!is_wp_error($post_id)) {
                // Set categories
                if (isset($post_data["categories"]) && is_array($post_data["categories"])) {
                    wp_set_post_categories($post_id, $post_data["categories"]);
                }
                
                // Set tags
                if (isset($post_data["tags"]) && is_array($post_data["tags"])) {
                    wp_set_post_tags($post_id, $post_data["tags"]);
                }
                
                $created[] = array(
                    "title" => $post_data["title"],
                    "id" => $post_id,
                    "success" => true
                );
            } else {
                $created[] = array(
                    "title" => $post_data["title"],
                    "error" => $post_id->get_error_message(),
                    "success" => false
                );
            }
        }
        
        return rest_ensure_response($created);
    }
    
    /**
     * Permission callbacks
     */
    public function get_permissions_check($request)
    {
        return current_user_can("edit_posts");
    }
    
    public function create_permissions_check($request)
    {
        return current_user_can("edit_posts") && current_user_can("publish_posts");
    }
} 