<?php
/**
 * Custom Post Types Loader
 *
 * Loads and registers all custom post types saved in WordPress options.
 * This file is executed on WordPress init to ensure all CPTs are registered.
 * Simplified version with fixed defaults and individual meta keys for each field.
 *
 * @package    WVC_Theme
 * @subpackage Bulk_Operations
 * @author     10Web
 * @since      1.0.0
 * @version    2.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class WVC_Custom_Post_Types_Loader
 *
 * Handles loading and registration of custom post types from saved configurations
 */
class WVC_Custom_Post_Types_Loader
{
    /**
     * Instance of this class
     *
     * @var WVC_Custom_Post_Types_Loader
     */
    private static $instance = null;

    /**
     * Get singleton instance
     *
     * @return WVC_Custom_Post_Types_Loader
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action('init', array($this, 'load_custom_post_types'));
    }

    /**
     * Load and register all custom post types from saved configurations
     */
    public function load_custom_post_types()
    {
        $cpt_configs = get_option('wvc_custom_post_types', array());

        if (empty($cpt_configs)) {
            return;
        }

        foreach ($cpt_configs as $post_type_key => $config) {
            $this->register_custom_post_type($post_type_key, $config);
        }
    }

    /**
     * Register a single custom post type
     *
     * @param string $post_type_key The post type key.
     * @param array  $config        The configuration array.
     */
private function register_custom_post_type($post_type_key, $config)
    {
        $post_type_label = $config['post_type_label'] ?? 'Items';
        $singular_label  = $config['singular_label'] ?? 'Item';

        // Register with fixed defaults
        $args = array(
            'labels'              => array(
                'name'               => $post_type_label,
                'singular_name'      => $singular_label,
                'add_new'            => sprintf(__('Add New %s', 'wvc-theme'), $singular_label),
                'add_new_item'       => sprintf(__('Add New %s', 'wvc-theme'), $singular_label),
                'edit_item'          => sprintf(__('Edit %s', 'wvc-theme'), $singular_label),
                'new_item'           => sprintf(__('New %s', 'wvc-theme'), $singular_label),
                'view_item'          => sprintf(__('View %s', 'wvc-theme'), $singular_label),
                'search_items'       => sprintf(__('Search %s', 'wvc-theme'), $post_type_label),
                'not_found'          => sprintf(__('No %s found', 'wvc-theme'), strtolower($post_type_label)),
                'not_found_in_trash' => sprintf(__('No %s found in Trash', 'wvc-theme'), strtolower($post_type_label)),
                'all_items'          => sprintf(__('All %s', 'wvc-theme'), $post_type_label),
            ),
            'public'              => true,
            'publicly_queryable'  => true,
            'show_ui'             => true,
            'show_in_menu'        => true,  // ✅ Visible in admin menu
            'query_var'           => true,
            'rewrite'             => array('slug' => $post_type_key),
            'capability_type'     => 'post',
            'has_archive'         => true,  // ✅ Fixed default
            'hierarchical'        => false, // ✅ Fixed default
            'menu_position'       => 20,    // After Pages
            'supports'            => array('title', 'editor', 'thumbnail', 'custom-fields'), // ✅ Fixed defaults + custom-fields for meta
            'show_in_rest'        => true,
        );

        register_post_type($post_type_key, $args);

        // Individual meta keys are registered in wvc-custom-post-type-rest-api.php
        // via the init_register_existing_metas() method
    }
}

// Initialize the Custom Post Types Loader
WVC_Custom_Post_Types_Loader::get_instance();
