<?php

namespace WVC\CartHandler;

/**
 * WVC Cart Handler
 *
 * Handles AJAX cart operations
 *
 * @package    WVC_Theme
 * @subpackage CartHandler
 * @author     10Web
 * @since      1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Detect which ecommerce plugin is active
 */
function wvc_detect_ecommerce_plugin() {
    if ( function_exists( 'WC' ) && class_exists( 'WooCommerce' ) ) {
        return 'woocommerce';
    }
    
    
    return false;
}

/**
 * Handle Add to Cart via AJAX
 */
function wvc_handle_ajax_add_to_cart() {
    // Debug: Log all received data
    error_log("WVC Debug - Add to Cart called with data: " . print_r($_POST, true));
    
    // Check if nonce exists
    $nonce_field = isset($_POST['nonce']) ? $_POST['nonce'] : 'MISSING';
    error_log("WVC Debug - Nonce field value: " . $nonce_field);
    
    // Verify nonce for security
    $nonce_valid = check_ajax_referer('wvc_nonce', 'nonce', false);
    error_log("WVC Debug - Nonce validation result: " . ($nonce_valid ? 'VALID' : 'INVALID'));
    
    if (!$nonce_valid) {
        error_log("Invalid nonce in Add to Cart submission - Expected action: 'wvc_nonce', Received nonce: " . $nonce_field);
        wp_send_json_success(array(
            'success' => false,
            'message' => 'Invalid security token'
        ));
        return;
    }

    // Debug received data
    error_log('Add to Cart Request: ' . print_r($_POST, true));

    // Get parameters
    $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
    $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;

    // Detect ecommerce platform
    $ecommerce_plugin = wvc_detect_ecommerce_plugin();
    
    if ( ! $ecommerce_plugin ) {
        wp_send_json_success(array(
            'success' => false,
            'message' => 'No supported ecommerce plugin is active'
        ));
        return;
    }

    try {
        switch ( $ecommerce_plugin ) {
            case 'woocommerce':
                $result = wvc_woocommerce_add_to_cart( $product_id, $quantity );
                break;
                
                
            default:
                wp_send_json_success(array(
                    'success' => false,
                    'message' => 'Ecommerce plugin not supported yet: ' . $ecommerce_plugin
                ));
                return;
        }
        
        if ( $result['success'] ) {
            wp_send_json_success(array(
                'success' => true,
                'message' => $result['data']['message'],
                'data' => $result['data']
            ));
        } else {
            wp_send_json_success(array(
                'success' => false,
                'message' => $result['data']['message']
            ));
        }
        
    } catch (\Exception $e) {
        wp_send_json_success(array(
            'success' => false,
            'message' => $e->getMessage()
        ));
    }
}

/**
 * Handle Remove from Cart via AJAX
 */
function wvc_handle_ajax_remove_from_cart() {
    // Verify nonce for security
    if (!check_ajax_referer('wvc_nonce', 'nonce', false)) {
        error_log("Invalid nonce in Remove from Cart submission");
        wp_send_json_error(array(
            'message' => 'Invalid security token'
        ), 403);
        return;
    }

    // Debug received data
    error_log('Remove from Cart Request: ' . print_r($_POST, true));

    $cart_item_key = isset($_POST['cart_item_key']) ? sanitize_text_field($_POST['cart_item_key']) : '';

    // Detect ecommerce platform
    $ecommerce_plugin = wvc_detect_ecommerce_plugin();
    
    if ( ! $ecommerce_plugin ) {
        wp_send_json_error( array( 'message' => 'No supported ecommerce plugin is active' ), 500 );
        return;
    }

    try {
        switch ( $ecommerce_plugin ) {
            case 'woocommerce':
                $result = wvc_woocommerce_remove_from_cart( $cart_item_key );
                break;
                
            
            default:
                wp_send_json_error( array( 'message' => 'Ecommerce plugin not supported yet: ' . $ecommerce_plugin ), 501 );
                return;
        }
        
        if ( $result['success'] ) {
            wp_send_json_success( $result['data'] );
        } else {
            wp_send_json_error( $result['data'], $result['status'] ?? 400 );
        }
        
    } catch (\Exception $e) {
        wp_send_json_error(array(
            'message' => $e->getMessage()
        ), 500);
    }
}

/**
 * Handle Get Cart State via AJAX
 */
function wvc_handle_ajax_get_cart_state() {
    // Verify nonce for security
    if (!check_ajax_referer('wvc_nonce', 'nonce', false)) {
        error_log("Invalid nonce in Get Cart State submission");
        wp_send_json_error(array(
            'message' => 'Invalid security token'
        ), 403);
        return;
    }

    // Debug received data
    error_log('Get Cart State Request: ' . print_r($_POST, true));

    // Detect ecommerce platform
    $ecommerce_plugin = wvc_detect_ecommerce_plugin();
    
    if ( ! $ecommerce_plugin ) {
        wp_send_json_error( array( 'message' => 'No supported ecommerce plugin is active' ), 500 );
        return;
    }

    try {
        switch ( $ecommerce_plugin ) {
            case 'woocommerce':
                $result = wvc_woocommerce_get_cart_state();
                break;
                
            
            default:
                wp_send_json_error( array( 'message' => 'Ecommerce plugin not supported yet: ' . $ecommerce_plugin ), 501 );
                return;
        }
        
        wp_send_json_success( $result );
        
    } catch (\Exception $e) {
        wp_send_json_error(array(
            'message' => $e->getMessage()
        ), 500);
    }
}

// =============================================================================
// WooCommerce Specific Functions
// =============================================================================

/**
 * WooCommerce: Add to Cart
 */
function wvc_woocommerce_add_to_cart( $product_id, $quantity ) {
    error_log("WVC Debug - Attempting to add product $product_id with quantity $quantity to cart");
    
    // Check if WooCommerce cart exists
    if ( ! WC()->cart ) {
        error_log("WVC Debug - WooCommerce cart not initialized");
        return array(
            'success' => false,
            'data' => array( 'message' => 'WooCommerce cart not initialized' )
        );
    }
    
    // Check if product exists and is purchasable
    $product = wc_get_product( $product_id );
    if ( ! $product ) {
        error_log("WVC Debug - Product $product_id not found");
        return array(
            'success' => false,
            'data' => array( 'message' => 'Product not found' )
        );
    }
    
    if ( ! $product->is_purchasable() ) {
        error_log("WVC Debug - Product $product_id is not purchasable");
        return array(
            'success' => false,
            'data' => array( 'message' => 'Product is not purchasable' )
        );
    }
    
    error_log("WVC Debug - Product validation passed, attempting to add to cart");
    $cart_item_key = WC()->cart->add_to_cart( $product_id, $quantity );
    error_log("WVC Debug - Cart add result: " . ($cart_item_key ? $cart_item_key : 'FALSE'));

    if ( $cart_item_key ) {
        error_log("WVC Debug - Successfully added to cart with key: $cart_item_key");
        return array(
            'success' => true,
            'data' => array(
                'message' => 'Item added to cart',
                'cart_hash' => WC()->cart->get_cart_hash(),
                'cart_count' => WC()->cart->get_cart_contents_count(),
                'cart_item_key' => $cart_item_key
            )
        );
    } else {
        error_log("WVC Debug - Failed to add to cart - checking WooCommerce notices");
        $notices = wc_get_notices('error');
        $error_message = 'Could not add to cart';
        if ( !empty($notices) ) {
            $error_message .= ': ' . implode(', ', array_column($notices, 'notice'));
        }
        return array(
            'success' => false,
            'data' => array( 'message' => $error_message )
        );
    }
}

/**
 * WooCommerce: Remove from Cart
 */
function wvc_woocommerce_remove_from_cart( $cart_item_key ) {
    if ( $cart_item_key && WC()->cart->get_cart_item( $cart_item_key ) ) {
        WC()->cart->remove_cart_item( $cart_item_key );
        return array(
            'success' => true,
            'data' => array(
                'message' => 'Item removed from cart',
                'cart_hash' => WC()->cart->get_cart_hash(),
                'cart_count' => WC()->cart->get_cart_contents_count()
            )
        );
    } else {
        return array(
            'success' => false,
            'status' => 404,
            'data' => array( 'message' => 'Item not found in cart' )
        );
    }
}

/**
 * WooCommerce: Get Cart State
 */
function wvc_woocommerce_get_cart_state() {
    $cart_data = array();

    foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
        $product = $cart_item['data']; // The WC_Product object
        
        // Get product image
        $image_id = $product->get_image_id();
        $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'thumbnail') : '';

        $cart_data[] = array(
            'key'       => $cart_item_key,
            'product_id'=> $cart_item['product_id'],
            'name'      => $product->get_name(),
            'quantity'  => $cart_item['quantity'],
            'price'     => $product->get_price(),
            'subtotal'  => WC()->cart->get_product_subtotal( $product, $cart_item['quantity'] ),
            'image'     => $image_url,
            'permalink' => $product->get_permalink()
        );
    }

    return array(
        'message' => 'Cart state retrieved',
        'cart' => $cart_data,
        'total' => WC()->cart->get_cart_total(),
        'count' => WC()->cart->get_cart_contents_count(),
        'hash' => WC()->cart->get_cart_hash(),
        'platform' => 'woocommerce'
    );
}

// =============================================================================
// Easy Digital Downloads Specific Functions
// =============================================================================

/**
 * EDD: Add to Cart
 */
function wvc_edd_add_to_cart( $product_id, $quantity ) {
    $options = array();
    $cart_item_key = edd_add_to_cart( $product_id, $options );

    if ( $cart_item_key !== false ) {
        return array(
            'success' => true,
            'data' => array(
                'message' => 'Item added to cart',
                'cart_count' => edd_get_cart_quantity(),
                'cart_item_key' => $cart_item_key
            )
        );
    } else {
        return array(
            'success' => false,
            'data' => array( 'message' => 'Could not add to cart' )
        );
    }
}

/**
 * EDD: Remove from Cart
 */
function wvc_edd_remove_from_cart( $cart_item_key ) {
    $cart_items = edd_get_cart_contents();
    $item_key = intval( $cart_item_key );
    
    if ( isset( $cart_items[ $item_key ] ) ) {
        edd_remove_from_cart( $item_key );
        return array(
            'success' => true,
            'data' => array(
                'message' => 'Item removed from cart',
                'cart_count' => edd_get_cart_quantity()
            )
        );
    } else {
        return array(
            'success' => false,
            'status' => 404,
            'data' => array( 'message' => 'Item not found in cart' )
        );
    }
}

/**
 * EDD: Get Cart State
 */
function wvc_edd_get_cart_state() {
    $cart_data = array();
    $cart_items = edd_get_cart_contents();

    foreach ( $cart_items as $key => $item ) {
        $product_id = $item['id'];
        $price = edd_get_download_price( $product_id );
        
        $cart_data[] = array(
            'key'       => $key,
            'product_id'=> $product_id,
            'name'      => get_the_title( $product_id ),
            'quantity'  => $item['quantity'],
            'price'     => $price,
            'subtotal'  => $price * $item['quantity'],
            'image'     => get_the_post_thumbnail_url( $product_id, 'thumbnail' ),
            'permalink' => get_permalink( $product_id )
        );
    }

    return array(
        'message' => 'Cart state retrieved',
        'cart' => $cart_data,
        'total' => edd_get_cart_total(),
        'count' => edd_get_cart_quantity(),
        'platform' => 'edd'
    );
}

// =============================================================================
// WP eCommerce Specific Functions
// =============================================================================

/**
 * WP eCommerce: Add to Cart (Basic implementation)
 */
function wvc_wp_ecommerce_add_to_cart( $product_id, $quantity ) {
    // Basic implementation - WP eCommerce has a complex cart system
    return array(
        'success' => false,
        'data' => array( 'message' => 'WP eCommerce integration not fully implemented yet' )
    );
}

/**
 * WP eCommerce: Remove from Cart
 */
function wvc_wp_ecommerce_remove_from_cart( $cart_item_key ) {
    return array(
        'success' => false,
        'data' => array( 'message' => 'WP eCommerce integration not fully implemented yet' )
    );
}

/**
 * WP eCommerce: Get Cart State
 */
function wvc_wp_ecommerce_get_cart_state() {
    return array(
        'message' => 'WP eCommerce integration not fully implemented yet',
        'cart' => array(),
        'total' => '0',
        'count' => 0,
        'platform' => 'wp_ecommerce'
    );
}

/**
 * Register AJAX actions for cart
 */
function wvc_register_cart_ajax_actions() {
    // Add to Cart
    add_action('wp_ajax_wvc_cart_add', __NAMESPACE__ . '\wvc_handle_ajax_add_to_cart');
    add_action('wp_ajax_nopriv_wvc_cart_add', __NAMESPACE__ . '\wvc_handle_ajax_add_to_cart');

    // Remove from Cart
    add_action('wp_ajax_wvc_cart_remove', __NAMESPACE__ . '\wvc_handle_ajax_remove_from_cart');
    add_action('wp_ajax_nopriv_wvc_cart_remove', __NAMESPACE__ . '\wvc_handle_ajax_remove_from_cart');

    // Get Cart State
    add_action('wp_ajax_wvc_cart_get', __NAMESPACE__ . '\wvc_handle_ajax_get_cart_state');
    add_action('wp_ajax_nopriv_wvc_cart_get', __NAMESPACE__ . '\wvc_handle_ajax_get_cart_state');
}

// Initialize the AJAX handlers
wvc_register_cart_ajax_actions();
