<?php
/**
 * WVC Site Manager
 *
 * @since      1.0.0
 * @package    WVC_Editor
 * @subpackage WVC_Editor/Managers
 */

// Exit if accessed directly
if ( ! defined('ABSPATH')) {
    exit;
}

/**
 * WVC Site Manager Class
 */
class WVC_Site_Manager
{
    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        $this->load_manager_classes();
    }

    /**
     * Reset site content
     *
     * @param bool $force Force permanent deletion
     * @param array $excluded Content types to exclude
     * @param array $plugin_slugs Plugin slugs to delete (optional)
     *
     * @return array Results of the reset operation
     */
    public function reset_site($force = false, $excluded = [], $plugin_slugs = [])
    {
        try {
            set_transient("wvc_reset_started", 1, 45);

            $results = [
                'posts'              => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'pages'              => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'products'           => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'menus'              => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'categories'         => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'tags'               => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'product_categories' => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'product_tags'       => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'media'              => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'plugins'            => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                'options'            => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
            ];

            // Delete Posts
            if ( ! in_array('posts', $excluded)) {
                $this->delete_posts($results, $force);
            } else {
                $results['posts']['skipped'] = true;
            }

            // Delete Pages
            if ( ! in_array('pages', $excluded)) {
                $this->delete_pages($results, $force);
            } else {
                $results['pages']['skipped'] = true;
            }

            // Delete WooCommerce Products
            if ( ! in_array('products', $excluded)) {
                $this->delete_products($results, $force);
            } else {
                $results['products']['skipped'] = true;
            }

            // Delete Menus
            if ( ! in_array('menus', $excluded)) {
                $this->delete_menus($results);
            } else {
                $results['menus']['skipped'] = true;
            }

            // Delete Categories (except "Uncategorized")
            if ( ! in_array('categories', $excluded)) {
                $this->delete_categories($results);
            } else {
                $results['categories']['skipped'] = true;
            }

            // Delete Tags
            if ( ! in_array('tags', $excluded)) {
                $this->delete_tags($results);
            } else {
                $results['tags']['skipped'] = true;
            }

            // Delete WooCommerce Product Categories
            if ( ! in_array('product_categories', $excluded)) {
                $this->delete_product_categories($results);
            } else {
                $results['product_categories']['skipped'] = true;
            }

            // Delete WooCommerce Product Tags
            if ( ! in_array('product_tags', $excluded)) {
                $this->delete_product_tags($results);
            } else {
                $results['product_tags']['skipped'] = true;
            }

            // Delete Media Attachments
            if ( ! in_array('media', $excluded)) {
                $this->delete_media($results, $force);
            } else {
                $results['media']['skipped'] = true;
            }

            // Delete Plugins (if plugin slugs provided)
            if ( ! empty($plugin_slugs)) {
                $this->delete_plugins($results, $plugin_slugs);
            } else {
                $results['plugins']['skipped'] = true;
            }

            // Delete Options
            if ( ! in_array('options', $excluded)) {
                $this->delete_options($results, $force);
            } else {
                $results['options']['skipped'] = true;
            }

            delete_transient("wvc_reset_started");

            return [
                'success'      => true,
                'message'      => 'Site reset completed successfully',
                'results'      => $results,
                'excluded'     => $excluded,
                'force_delete' => $force
            ];

        } catch (Exception $e) {
            delete_transient("wvc_reset_started");

            return [
                'success' => false,
                'message' => 'Site reset failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Delete WordPress posts
     */
    private function delete_posts(&$results, $force)
    {
        $posts        = get_posts(['post_type' => 'post', 'numberposts' => -1, 'post_status' => 'any']);
        $post_manager = WVC_Post_Manager::get_instance();
        foreach ($posts as $post) {
            $result = $post_manager->delete_post($post->ID, $force);
            if (is_wp_error($result)) {
                $results['posts']['failed']++;
            } else {
                $results['posts']['deleted']++;
            }
        }
    }

    /**
     * Delete WordPress pages (only WVC editor pages with _wvc_editor meta)
     */
    private function delete_pages(&$results, $force)
    {
        $pages = get_posts([
            'post_type'   => 'page',
            'numberposts' => -1,
            'post_status' => 'any',
            /*'meta_query'  => [
                [
                    'key'     => '_wvc_editor',
                    'compare' => 'EXISTS'
                ]
            ]*/
        ]);

        $page_manager = WVC_Page_Manager::get_instance();
        foreach ($pages as $page) {
            $result = $page_manager->delete_page($page->ID, $force);
            if (is_wp_error($result)) {
                $results['pages']['failed']++;
            } else {
                $results['pages']['deleted']++;
            }
        }
    }

    /**
     * Delete WooCommerce products
     */
    private function delete_products(&$results, $force)
    {
        $products     = get_posts(['post_type' => 'product', 'numberposts' => -1, 'post_status' => 'any']);
        $post_manager = WVC_Post_Manager::get_instance();
        foreach ($products as $product) {
            $result = $post_manager->delete_post($product->ID, $force);
            if (is_wp_error($result)) {
                $results['products']['failed']++;
            } else {
                $results['products']['deleted']++;
            }
        }
    }

    /**
     * Delete navigation menus
     */
    private function delete_menus(&$results)
    {
        $menus        = wp_get_nav_menus();
        $menu_manager = WVC_Menu_Manager::get_instance();
        foreach ($menus as $menu) {
            $result = $menu_manager->delete_menu($menu->term_id);
            if (is_wp_error($result)) {
                $results['menus']['failed']++;
            } else {
                $results['menus']['deleted']++;
            }
        }
    }

    /**
     * Delete WordPress categories (except "Uncategorized")
     */
    private function delete_categories(&$results)
    {
        $categories   = get_categories(['hide_empty' => false]);
        $post_manager = WVC_Post_Manager::get_instance();

        foreach ($categories as $category) {
            // Skip the default "Uncategorized" category
            if ($category->slug === 'uncategorized') {
                continue;
            }
            $result = $post_manager->delete_category($category->term_id);
            if (is_wp_error($result)) {
                $results['categories']['failed']++;
            } else {
                $results['categories']['deleted']++;
            }
        }
    }

    /**
     * Delete WordPress tags
     */
    private function delete_tags(&$results)
    {
        $tags         = get_tags(['hide_empty' => false]);
        $post_manager = WVC_Post_Manager::get_instance();
        foreach ($tags as $tag) {
            $result = $post_manager->delete_tag($tag->term_id);
            if (is_wp_error($result)) {
                $results['tags']['failed']++;
            } else {
                $results['tags']['deleted']++;
            }
        }
    }

    /**
     * Delete WooCommerce product categories
     */
    private function delete_product_categories(&$results)
    {
        $product_categories = get_terms(['taxonomy' => 'product_cat', 'hide_empty' => false]);
        foreach ($product_categories as $category) {
            // Skip the default "Uncategorized" category
            if ($category->slug === 'uncategorized') {
                continue;
            }
            $result = wp_delete_term($category->term_id, 'product_cat');
            if (is_wp_error($result)) {
                $results['product_categories']['failed']++;
            } else {
                $results['product_categories']['deleted']++;
            }
        }
    }

    /**
     * Delete WooCommerce product tags
     */
    private function delete_product_tags(&$results)
    {
        $product_tags = get_terms(['taxonomy' => 'product_tag', 'hide_empty' => false]);
        foreach ($product_tags as $tag) {
            $result = wp_delete_term($tag->term_id, 'product_tag');
            if (is_wp_error($result)) {
                $results['product_tags']['failed']++;
            } else {
                $results['product_tags']['deleted']++;
            }
        }
    }

    /**
     * Delete media attachments
     */
    private function delete_media(&$results, $force)
    {
        $media_items = get_posts(['post_type' => 'attachment', 'numberposts' => -1, 'post_status' => 'any']);
        foreach ($media_items as $media) {
            $result = wp_delete_attachment($media->ID, $force);
            if (is_wp_error($result) || $result === false) {
                $results['media']['failed']++;
            } else {
                $results['media']['deleted']++;
            }
        }
    }

    /**
     * Delete plugins
     */
    private function delete_plugins(&$results, $plugin_slugs)
    {
        // Load Plugin Manager
        if ( ! class_exists('WVC_Plugin_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/plugin-manager.php';
        }

        $plugin_manager = WVC_Plugin_Manager::get_instance();

        foreach ($plugin_slugs as $plugin_slug) {
            $plugin_slug = sanitize_text_field($plugin_slug);
            $result      = $plugin_manager->uninstall_plugin($plugin_slug);

            if ( ! $result['success']) {
                $results['plugins']['failed']++;
            } else {
                $results['plugins']['deleted']++;
            }
        }
    }

    /**
     * Delete WordPress options
     */
    private function delete_options(&$results, $force)
    {
        $options_to_delete = ["page_on_front"];
        foreach ($options_to_delete as $option) {
            delete_option($option);
            $results['options']['deleted']++;
        }
    }

    /**
     * Load required manager classes
     */
    private function load_manager_classes()
    {
        // Menu Manager
        if ( ! class_exists('WVC_Menu_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/menu-manager.php';
        }

        // Page Manager
        if ( ! class_exists('WVC_Page_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/page-manager.php';
        }

        // Post Manager
        if ( ! class_exists('WVC_Post_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/post-manager.php';
        }
    }
}