<?php
/**
 * Term Manager Class
 *
 * Provides generic CRUD for WordPress terms across any taxonomy.
 *
 * @package    WVC_Theme
 * @subpackage Content_Managers
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
	exit;
}

/**
 * Class WVC_Term_Manager
 *
 * Generic term CRUD helpers for any taxonomy.
 */
class WVC_Term_Manager {

	/**
	 * Instance of this singleton class
	 *
	 * @var WVC_Term_Manager
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return WVC_Term_Manager
	 */
	public static function get_instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Private constructor to prevent direct instantiation
	}

	/**
	 * Create a term in the given taxonomy
	 *
	 * @param string $taxonomy Taxonomy name
	 * @param array  $args     Term args: name (required), slug, description, parent
	 * @return array|WP_Error  Term data array on success, WP_Error on failure
	 */
	public function create_term( $taxonomy, $args ) {
		$defaults = array(
			"description" => "",
			"parent" => 0,
			"slug" => "",
		);
		$args = wp_parse_args( $args, $defaults );

		if ( empty( $args["name"] ) ) {
			return new WP_Error(
				"missing_name",
				__( "Term name is required", "wvc-theme" ),
				array( "status" => 400 )
			);
		}

		// Try to find existing by name
		$existing_by_name = get_term_by( "name", $args["name"], $taxonomy );
		if ( $existing_by_name && ! is_wp_error( $existing_by_name ) ) {
			return $this->get_term( $existing_by_name->term_id, $taxonomy );
		}

		// Try to find existing by slug if provided
		if ( ! empty( $args["slug"] ) ) {
			$existing_by_slug = get_term_by( "slug", $args["slug"], $taxonomy );
			if ( $existing_by_slug && ! is_wp_error( $existing_by_slug ) ) {
				return $this->get_term( $existing_by_slug->term_id, $taxonomy );
			}
		}

		$result = wp_insert_term(
			$args["name"],
			$taxonomy,
			array(
				"description" => $args["description"],
				"parent" => (int) $args["parent"],
				"slug" => $args["slug"],
			)
		);

		if ( is_wp_error( $result ) ) {
			if ( $result->get_error_code() === "term_exists" ) {
				$term_id = $result->get_error_data();
				if ( is_array( $term_id ) ) {
					$term_id = isset( $term_id["term_id"] ) ? $term_id["term_id"] : 0;
				}
				if ( $term_id ) {
					return $this->get_term( $term_id, $taxonomy );
				}
			}
			return $result;
		}

		$term_id = (int) $result["term_id"];

		$this->maybe_update_term_uid( $term_id, $args );

		return $this->get_term( $term_id, $taxonomy );
	}

	/**
	 * Get a term by ID and taxonomy
	 *
	 * @param int    $term_id  Term ID
	 * @param string $taxonomy Taxonomy name
	 * @return array|WP_Error  Term data array on success, WP_Error on failure
	 */
	public function get_term( $term_id, $taxonomy ) {
		$term = get_term( $term_id, $taxonomy );
		if ( ! $term || is_wp_error( $term ) ) {
			return new WP_Error(
				"term_not_found",
				__( "Term not found", "wvc-theme" ),
				array( "status" => 404 )
			);
		}

		$data = array(
			"id" => $term->term_id,
			"name" => str_replace( "&amp;", "&", $term->name ),
			"slug" => $term->slug,
			"description" => str_replace( "&amp;", "&", $term->description ),
			"parent" => (int) $term->parent,
			"count" => (int) $term->count,
			"taxonomy" => $taxonomy,
			"link" => get_term_link( $term ),
		);

		$meta = get_term_meta( $term->term_id );
		if ( $meta ) {
			$formatted_meta = array();
			foreach ( $meta as $key => $values ) {
				$formatted_meta[ $key ] = maybe_unserialize( $values[0] );
			}
			$data["meta"] = $formatted_meta;
		}

		return $data;
	}

	/**
	 * Update a term
	 *
	 * @param int    $term_id  Term ID
	 * @param string $taxonomy Taxonomy name
	 * @param array  $args     name, slug, description, parent, meta
	 * @return array|WP_Error
	 */
	public function update_term( $term_id, $taxonomy, $args ) {
		$term = get_term( $term_id, $taxonomy );
		if ( ! $term || is_wp_error( $term ) ) {
			return new WP_Error(
				"term_not_found",
				__( "Term not found", "wvc-theme" ),
				array( "status" => 404 )
			);
		}

		$update_args = array();
		if ( isset( $args["name"] ) ) {
			$update_args["name"] = $args["name"];
		}
		if ( isset( $args["slug"] ) ) {
			$update_args["slug"] = $args["slug"];
		}
		if ( isset( $args["description"] ) ) {
			$update_args["description"] = $args["description"];
		}
		if ( isset( $args["parent"] ) ) {
			$update_args["parent"] = (int) $args["parent"];
		}

		if ( ! empty( $update_args ) ) {
			$result = wp_update_term( $term_id, $taxonomy, $update_args );
			if ( is_wp_error( $result ) ) {
				return $result;
			}
		}

		if ( isset( $args["meta"] ) && is_array( $args["meta"] ) ) {
			foreach ( $args["meta"] as $meta_key => $meta_value ) {
				update_term_meta( $term_id, $meta_key, $meta_value );
			}
		}

		$this->maybe_update_term_uid( $term_id, $args );

		return $this->get_term( $term_id, $taxonomy );
	}

	/**
	 * Delete a term
	 *
	 * @param int    $term_id  Term ID
	 * @param string $taxonomy Taxonomy name
	 * @return array|WP_Error
	 */
	public function delete_term( $term_id, $taxonomy ) {
		$term = get_term( $term_id, $taxonomy );
		if ( ! $term || is_wp_error( $term ) ) {
			return new WP_Error(
				"term_not_found",
				__( "Term not found", "wvc-theme" ),
				array( "status" => 404 )
			);
		}

		$result = wp_delete_term( $term_id, $taxonomy );
		if ( ! $result || is_wp_error( $result ) ) {
			if ( is_wp_error( $result ) ) {
				return $result;
			}
			return new WP_Error(
				"delete_failed",
				__( "Failed to delete term", "wvc-theme" ),
				array( "status" => 500 )
			);
		}

		return array(
			"deleted" => true,
			"term_id" => $term_id,
			"taxonomy" => $taxonomy,
		);
	}

	/**
	 * Store or remove term UID meta based on args.
	 *
	 * @param int   $term_id Term ID.
	 * @param array $args    Input arguments possibly containing uid.
	 */
	private function maybe_update_term_uid( $term_id, $args ) {
		if ( ! isset( $args["uid"] ) ) {
			return;
		}

		$uid = trim( (string) $args["uid"] );

		if ( $uid === "" ) {
			return;
		}

		update_term_meta( $term_id, "wvc_uid", sanitize_text_field( $uid ) );
	}
}


