<?php

namespace WVC\Core;

/**
 * WVC Theme Core Functions
 *
 * This file contains core functionality for the WVC theme including
 * post meta registration and other essential features.
 *
 * @package    WVC_Theme
 * @subpackage Core
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Core Class
 *
 * Handles core functionality for the WVC theme including
 * post meta registration and other essential features.
 */
class Core
{

    /**
     * Meta key for WVC editor
     */
    const WVC_EDITOR_META_KEY = '_wvc_editor';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Hook the registration function to init
        add_action("init", array($this, "register_post_meta"));

        // AJAX handlers
        add_action("wp_ajax_wvc_ajax", array($this, "handle_ajax_request"));
        add_action("wp_ajax_nopriv_wvc", array($this, "handle_ajax_request"));
    }

    /**
     * Register post meta for page types
     *
     * Registers the wvc_editor meta field as a boolean with default value of false
     */
    public function register_post_meta()
    {
        register_post_meta(
            "page", // Object type - pages only
            self::WVC_EDITOR_META_KEY, // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );

        register_post_meta(
            "page", // Object type - pages only
            "_wvc_home", // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );

        register_post_meta(
            "page", // Object type - pages only
            "_wvc_checkout", // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );
        register_post_meta(
            "page", // Object type - pages only
            "_wvc_cart", // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );

		// WooCommerce product UID meta
		register_post_meta(
			"product", // Object type - WooCommerce products
			"wvc_uid", // Meta key
			array(
				"type"          => "string",
				"single"        => true,
				"default"       => "",
				"show_in_rest"  => false,
				"auth_callback" => function () {
					return current_user_can("edit_products");
				}
			)
		);

        
    }

    /**
     * Get WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     *
     * @return bool WVC editor meta value
     */
    public static function get_wvc_editor_meta($post_id)
    {
        return (bool)get_post_meta($post_id, self::WVC_EDITOR_META_KEY, true);
    }

    public static function get_vibecode_singular_template($type = "single", $post_type = "post")
    {
        return \WVC\Includes\Theme\Theme_Core::is_there_a_vibecode_template($type, $post_type);
    }

    public static function is_vibecode_singular()
    {
        return \WVC\Includes\Theme\Theme_Core::is_vibecode_singular();
    }
    public static function get_vibecode_archive_template(){
        # check if is archive
        if(is_archive()){
            return \WVC\Includes\Theme\Theme_Core::is_there_a_vibecode_template("archive", get_post_type());
        }
        
        return false;
        
    }

    /**
     * Set WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     * @param bool $value WVC editor meta value
     *
     * @return bool|int Meta ID if the key didn't exist, true on successful update, false on failure
     */
    public static function set_wvc_editor_meta($post_id, $value)
    {
        return update_post_meta($post_id, self::WVC_EDITOR_META_KEY, (bool)$value);
    }

    /**
     * Delete WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     *
     * @return bool True on successful delete, false on failure
     */
    public static function delete_wvc_editor_meta($post_id)
    {
        return delete_post_meta($post_id, self::WVC_EDITOR_META_KEY);
    }

    /**
     * Check if a post is built with WVC editor
     *
     * @param int $post_id Post ID
     *
     * @return bool True if built with WVC editor, false otherwise
     */
    public static function is_built_with_wvc($post_id)
    {
        return self::get_wvc_editor_meta($post_id);
    }


    public static function is_empty_vibecode_site()
    {
        // Check if wvc_empty option is set to 1
        $wvc_empty = get_option("wvc_empty", 0);

        if ($wvc_empty != 1) {
            return false;
        }

        // Check if there are no published posts with WVC editor meta key
        $published_posts = get_posts(array(
            'post_status'  => 'publish',
            'meta_key'     => self::WVC_EDITOR_META_KEY,
            'meta_value'   => true,
            'meta_compare' => '=',
            'numberposts'  => 1,
            'fields'       => 'ids'
        ));

        if ( ! empty($published_posts)) {
            return false;
        }

        return true;
    }


    /**
     * Mark the post as "built with WVC" or not
     *
     * @param int $post_id Post ID
     * @param bool $is_built_with_wvc Whether the post is built with WVC
     *
     * @return bool|int Meta ID if the key didn't exist, true on successful update, false on failure
     */
    public static function set_is_built_with_wvc($post_id, $is_built_with_wvc)
    {
        if ($is_built_with_wvc) {
            return self::set_wvc_editor_meta($post_id, true);
        } else {
            return self::delete_wvc_editor_meta($post_id);
        }
    }

    /**
     * Get 10Web-related configuration values
     *
     * @return array Array containing access token, user ID, domain ID, and workspace ID
     */
    public static function get_tenweb_config()
    {
        $config = array(
            'accessToken' => get_site_option('tenweb_access_token'),
            'wpUserId'    => get_current_user_id(),
            'domainId'    => get_site_option('tenweb_domain_id'),
            'workspaceId' => get_site_option('tenweb_workspace_id')
        );

        $tenweb_user_info = get_site_option('tenweb_user_info');
        if ($tenweb_user_info && isset($tenweb_user_info['agreement_info']['clients_id'])) {
            $config["clientId"] = $tenweb_user_info['agreement_info']['clients_id'];
        } else {
            $config["clientId"] = null;
        }

        return $config;
    }


    /**
     * Get WVC API URL with debug override support
     *
     * @return string|null The WVC API URL
     */
    public static function get_wvc_api_url()
    {
        // Check if debug mode is enabled and get the debug config
        $debug_mode = get_transient('wvc_debug_mode_enabled');
        if ($debug_mode) {
            $debug_configs = get_option('wvc_debug_configs', array());
            if ( ! empty($debug_configs['api_url'])) {
                return $debug_configs['api_url'] . "/api";
            }
        }

        // Fall back to default URL
        if (defined('TENWEB_WVC_URL')) {
            return TENWEB_WVC_URL . "/api";
        }

        return null;
    }

    /**
     * Convert WordPress queried object to React component query format
     *
     * Converts the result of get_queried_object() into a query format
     * that matches what the React EcommerceProductsProvider expects in wp_query prop.
     *
     * @param mixed $wp_query_obj The queried object (WP_Post, WP_Term, WP_Post_Type, WP_User, etc.)
     * @return array Query parameters compatible with WordPress REST API and React component
     */
    public static function convert_wp_query_to_react_query($wp_query_obj)
    {
        // Default query parameters
        $default_posts_per_page = (int) get_option('posts_per_page', 10);
        $query = array(
            'status' => 'publish',
            'per_page' => $default_posts_per_page,
            'page' => 1,
        );

        // If no queried object, return default query
        if (empty($wp_query_obj)) {
            return $query;
        }

        // Check if it's a WP_Post object (has post_type, ID, post_name, etc.)
        if ($wp_query_obj instanceof \WP_Post) {
            // Single post query
            $query['id'] = intval($wp_query_obj->ID);

            if (!empty($wp_query_obj->post_name)) {
                $query['slug'] = sanitize_text_field($wp_query_obj->post_name);
            }
            if (!empty($wp_query_obj->post_type)) {
                $query['post_type'] = sanitize_text_field($wp_query_obj->post_type);
            }
            if (!empty($wp_query_obj->post_status)) {
                $query['status'] = sanitize_text_field($wp_query_obj->post_status);
            }

        }

        // Check if it's a WP_Term object (has taxonomy, term_id, slug, etc.)
        if ($wp_query_obj instanceof \WP_Term) {
            $taxonomy = !empty($wp_query_obj->taxonomy) ? sanitize_text_field($wp_query_obj->taxonomy) : '';
            $term_id  = !empty($wp_query_obj->term_id) ? intval($wp_query_obj->term_id) : 0;

            if (!empty($taxonomy) && $term_id > 0) {
                // Query posts by taxonomy term
                // WordPress REST API uses taxonomy slug as parameter (e.g., 'product_cat', 'product_tag')
                $query[$taxonomy] = $term_id;
                
                // Try to determine post_type from taxonomy
                $taxonomy_obj = get_taxonomy($taxonomy);
                if ($taxonomy_obj && !empty($taxonomy_obj->object_type)) {
                    // Use the first post type associated with this taxonomy
                    $query['post_type'] = sanitize_text_field($taxonomy_obj->object_type[0]);
                }
            }
            // Don't return here - continue to extract query parameters from global query
        }
        // Check if it's a WP_Post_Type object (has name, etc.)
        elseif ($wp_query_obj instanceof \WP_Post_Type) {
            // Archive query for a post type
            if (!empty($wp_query_obj->name)) {
                $query['post_type'] = sanitize_text_field($wp_query_obj->name);
            }
            // Don't return here - continue to extract query parameters from global query
        }
        // Check if it's a WP_User object (has ID, user_login, etc.)
        elseif ($wp_query_obj instanceof \WP_User) {
            // Author archive query
            $query['author'] = intval($wp_query_obj->ID);
            // Don't return here - continue to extract query parameters from global query
        }

        // Extract query parameters (orderby, order, per_page, etc.) from global query
        // This applies to all cases: WP_Term, WP_Post_Type, WP_User, and fallback cases
        global $wp_query;
        if ($wp_query) {
            // Handle post_type (if not already set from queried object)
            if (empty($query['post_type']) && isset($wp_query->query_vars['post_type'])) {
                $query['post_type'] = sanitize_text_field($wp_query->query_vars['post_type']);
            }

            // Handle pagination from global query
            if (isset($wp_query->query_vars['paged']) && $wp_query->query_vars['paged'] > 0) {
                $query['page'] = intval($wp_query->query_vars['paged']);
            }

            // Handle per_page/posts_per_page
            // Check both query_vars (WordPress standard) and URL parameter (REST API standard)
            if (isset($wp_query->query_vars['posts_per_page']) && $wp_query->query_vars['posts_per_page'] > 0) {
                $query['per_page'] = intval($wp_query->query_vars['posts_per_page']);
            } elseif (isset($_GET['per_page']) && intval($_GET['per_page']) > 0) {
                $query['per_page'] = intval($_GET['per_page']);
            }

            // Handle orderby - map WordPress orderby values to REST API compatible values
            if (isset($wp_query->query_vars['orderby']) && !empty($wp_query->query_vars['orderby'])) {
                $orderby = sanitize_text_field($wp_query->query_vars['orderby']);
                
                // Map WordPress orderby values to REST API compatible values
                $orderby_mapping = array(
                    'post_title'     => 'title',
                    'title'          => 'title',
                    'ID'             => 'id',
                    'post_id'        => 'id',
                    'post_date'      => 'date',
                    'date'           => 'date',
                    'post_modified'   => 'modified',
                    'modified'       => 'modified',
                    'post_name'      => 'slug',
                    'name'           => 'slug',
                    'post_author'     => 'author',
                    'author'         => 'author',
                    'post_parent'    => 'parent',
                    'parent'         => 'parent',
                    'menu_order'     => 'menu_order', // Only valid for pages/hierarchical post types
                );
                
                // Check if we have a mapping for this orderby value
                if (isset($orderby_mapping[$orderby])) {
                    $mapped_orderby = $orderby_mapping[$orderby];
                    
                    // Valid REST API orderby values
                    $valid_orderby = array('author', 'date', 'id', 'include', 'modified', 'parent', 'relevance', 'slug', 'include_slugs', 'title', 'menu_order');
                    
                    // Only add if it's a valid REST API value
                    if (in_array($mapped_orderby, $valid_orderby, true)) {
                        $query['orderby'] = $mapped_orderby;
                    }
                } elseif (in_array($orderby, array('author', 'date', 'id', 'include', 'modified', 'parent', 'relevance', 'slug', 'include_slugs', 'title', 'menu_order'), true)) {
                    // Already a valid REST API value
                    $query['orderby'] = $orderby;
                }
                // If orderby is not valid/mappable, don't include it (will use REST API default)
            }

            // Handle order
            if (isset($wp_query->query_vars['order']) && !empty($wp_query->query_vars['order'])) {
                $order = strtolower(sanitize_text_field($wp_query->query_vars['order']));
                // REST API only accepts 'asc' or 'desc'
                if (in_array($order, array('asc', 'desc'), true)) {
                    $query['order'] = $order;
                }
            }

            // Handle search query
            if (isset($wp_query->query_vars['s']) && !empty($wp_query->query_vars['s'])) {
                $query['search'] = sanitize_text_field($wp_query->query_vars['s']);
            }
        }

        return $query;
    }

    public static function get_wvc_params_to_localize($post_id, $term_id, $taxonomy, $is_editor = false)
    {

        $wvc_url = self::get_wvc_api_url();

        $wp_query = get_queried_object();

        // Convert queried object to React component query format
        $react_query = self::convert_wp_query_to_react_query($wp_query);
        $postType = $post_id ? get_post_type($post_id) : '';

        $params =  array_merge(array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'resturl' => rest_url(),
            'nonce' => wp_create_nonce('wvc_nonce'),
            'postId' => $post_id,
            'termId' => $term_id,
            "postType" => $postType,
            'taxonomy' => $taxonomy,
            'wpQuery' => $react_query,
            'postTitle' => $post_id ? get_the_title($post_id) : '',
            'postStatus' => $post_id ? get_post_status($post_id) : '',
            'isBuiltWithWvc' => $post_id ? \WVC\Core\Core::get_wvc_editor_meta($post_id) : false,
            'adminUrl'       => admin_url(),
            'wvcUrl'         => $wvc_url,
            "homeUrl"        => get_home_url(),
            'jsUtilsLink'    => get_template_directory_uri() . "/assets/js/index.umd.js",
            "siteTitle"      => get_bloginfo('name'),
            "companyName"    => TENWEB_COMPANY_NAME,
            "whitelabelLogo" => wvc_get_whitelabel_image(''),
            "dashboardUrl"   => TENWEB_DASHBOARD,
            "coreUrl"        => parse_url(TENWEB_SIGNUP_MAGIC_LINK_URL,
                    PHP_URL_SCHEME) . '://' . parse_url(TENWEB_SIGNUP_MAGIC_LINK_URL, PHP_URL_HOST)

        ), \WVC\Core\Core::get_tenweb_config());

        if ( ! $is_editor) {
            $params['nonce']        = null;
            $params['adminUrl']     = null;
            $params['dashboardUrl'] = null;
            $params['coreUrl']      = null;
            $params['accessToken']  = null;
            $params['wpUserId']     = null;
            $params['domainId']     = null;
            $params['workspaceId']  = null;
            $params['clientId']     = null;

        }

        return $params;

    }

    /**
     * AJAX handler for React components
     */
    public function handle_ajax_request()
    {
        check_ajax_referer("wvc_nonce", "nonce");

        $action = sanitize_text_field($_POST["action_type"]);

        switch ($action) {
            case "get_posts":
                $posts = get_posts(
                    array(
                        "numberposts" => 10,
                        "post_status" => "publish"
                    )
                );
                wp_send_json_success($posts);
                break;
            case "send_ga_event":
                $analytics_data = $_POST["data"];
                if (defined('TENWEB_SEND_GA_EVENT')) {
                    // Decode the escaped JSON string to get proper JSON
                    $decoded_data = stripslashes($analytics_data);
                    $body         = json_decode($decoded_data, true);
                    if ( ! is_array($body)) {
                        $body = array("data" => $decoded_data);
                    }
                    $tenweb_config = \WVC\Core\Core::get_tenweb_config();
                    unset($tenweb_config['accessToken']);

                    // Transform field names for GA event

                    $transformed_config = array();
                    if (isset($tenweb_config['domainId'])) {
                        $transformed_config['website_id'] = $tenweb_config['domainId'];
                    }
                    if (isset($tenweb_config['wpUserId'])) {
                        $transformed_config['wp_user_id'] = $tenweb_config['wpUserId'];
                    }
                    if (isset($tenweb_config['clientId'])) {
                        $transformed_config['tenweb_user_id'] = $tenweb_config['clientId'];
                    }
                    if (isset($tenweb_config['workspaceId'])) {
                        $transformed_config['workspace_id'] = $tenweb_config['workspaceId'];
                    }

                    if (is_array($body)) {

                        $body = array_merge($body, $transformed_config);
                    } else {
                        $body = $transformed_config;
                    }
                    $response = wp_remote_post(TENWEB_SEND_GA_EVENT, array(
                        'body'    => $body,
                        'headers' => array(
                            'Accept'       => 'application/x.10webcore.v1+json',
                            "Content-Type" => "application/x-www-form-urlencoded"
                        )
                    ));
                    if ( ! is_wp_error($response)) {
                        $response_code = wp_remote_retrieve_response_code($response);
                        if ($response_code >= 200 && $response_code < 300) {
                            wp_send_json_success();
                        } else {
                            $response_body = wp_remote_retrieve_body($response);
                            wp_send_json_error("HTTP {$response_code}: {$response_body}");
                        }
                    } else {
                        wp_send_json_error($response->get_error_message());
                    }

                }
                wp_send_json_error("GA endpoint is not defined");
                break;
            case "upload_media":
                $data = $_POST["data"];
                $decoded_data = stripslashes($data);
                $body  = json_decode($decoded_data, true);
                if (empty($body['base64_data']) || empty($body['filename']) || empty($body['mime_type'])) {
                    wp_send_json_error("Empty base64_data or filename or mime_type");
                }

                require_once get_template_directory() . '/includes/content-managers/post-manager.php';
                $post_manager = \WVC_Post_Manager::get_instance();

                $base64_data  = $body['base64_data'];
                $filename     = $body['filename'];
                $content_type = $body['mime_type'];
                // Decode the base64 data
                $file_data = base64_decode($base64_data);
                // Make sure we have a valid filename
                $result = $post_manager->upload_media(
                    $file_data,
                    $filename,
                    $content_type
                );
                wp_send_json_success($result);
                break;
            case "remove_media":
                $data = $_POST["data"];
                $decoded_data = stripslashes($data);
                $body  = json_decode($decoded_data, true);
                if (empty($body['id'])) {
                    wp_send_json_error("Empty id");
                }

                require_once get_template_directory() . '/includes/content-managers/post-manager.php';
                $post_manager = \WVC_Post_Manager::get_instance();

                $result = $post_manager->delete_media($body['id']);
                wp_send_json_success($result);
                break;
            default:
                wp_send_json_error("Invalid action");
        }
    }

    public static function get_menus_to_localize($post_id, $is_editor = false)
    {
        // Build a localized structure mirroring REST API responses
        if ( ! class_exists('WVC_Menu_Manager')) {
            return array(
                'menus'         => array(),
                'menusDetailed' => array(),
            );
        }
        $menu_manager = \WVC_Menu_Manager::get_instance();
        // Matches REST: GET /menus
        $menus = $menu_manager->get_menus();
        // Matches REST: GET /menus/{id} for each menu
        $menus_detailed = array();
        if (is_array($menus)) {
            foreach ($menus as $menu_summary) {
                if (is_array($menu_summary) && isset($menu_summary['id'])) {
                    $menu_full = $menu_manager->get_menu($menu_summary['id']);
                    if ( ! is_wp_error($menu_full)) {
                        $menus_detailed[] = $menu_full;
                    }
                }
            }
        }

        return array(
            'menus'         => $menus,
            'menusDetailed' => $menus_detailed,
        );
    }
}

// Initialize the core functionality
new Core();
