<?php

namespace WVC\Editor;

/**
 * Admin
 *
 * @package WVC_Theme
 * @subpackage Editor
 * @author 10Web
 * @since 1.0.0
 * @version 1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Admin Class
 * 
 * Handles admin-related functionalities for WVC editor
 * This class manages admin lists, row actions, bulk actions, and post states
 */
class Admin {
    

    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Hook into page row actions
        add_filter("page_row_actions", array($this, "customize_page_row_actions"), 10, 2);
        add_filter("post_row_actions", array($this, "customize_page_row_actions"), 10, 2);

        add_filter("get_edit_post_link", array($this, "customize_edit_post_link"), 10, 3);
        

        

        
        // Hook into admin body class filter
        add_filter('admin_body_class', array($this, 'body_status_classes'));
                
        // Hook into title display to add WVC indicator
        add_filter('the_title', array($this, 'modify_title_for_wvc_pages'), 10, 2);
    }
    
    
    
    /**
     * Customize page row actions in admin
     * 
     * Adds custom edit link for pages built with WVC editor
     * Removes regular edit link when built with WVC
     * 
     * @param array $actions Row actions
     * @param \WP_Post $post Post object
     * @return array Modified actions
     */
    public function customize_page_row_actions($actions, $post) {
        // Only apply to pages
        if ($post->post_type !== "page" && $post->post_type !== 'post' && $post->post_type !== 'product') {
            return $actions;
        }
        
        // Check if page is built with WVC editor
        $is_built_with_wvc = $this->is_built_with_wvc($post);
        


        // If built with WVC, remove the regular edit link
        if ($is_built_with_wvc) {
            unset($actions["edit"]);
            
            // If it's also a draft with empty content, remove preview action
            if ($post->post_status === "draft" && empty(trim($post->post_content))) {
                unset($actions["view"]);
            }
        }

        if (($post->post_type === 'post' or $post->post_type === 'product' )&& isset($actions["edit"])) {
            // For posts not built with WVC, rename "edit" to "Edit Content"
            // Extract the URL from the existing action and rebuild with new label
            preg_match('/href=["\']([^"\']+)["\']/', $actions["edit"], $matches);
            if (!empty($matches[1])) {
                $edit_url = $matches[1];
                $title = get_the_title($post->ID) ?: __('(no title)', 'wvc-theme');
                $actions["edit"] = sprintf(
                    '<a href="%s" aria-label="%s">%s</a>',
                    esc_url($edit_url),
                    esc_attr(sprintf(__('Edit Content &#8220;%s&#8221;', 'wvc-theme'), $title)),
                    esc_html__('Edit Content', 'wvc-theme')
                );
            }
        }


        
        // Add custom edit link for pages built with WVC
        if ($is_built_with_wvc || $post->post_type === 'post' || $post->post_type === 'product' ) {
            // Create editor instance to get the URL
            $editor = new \WVC\Editor\Editor($post);
            $custom_edit_url = $editor->get_edit_url();
            $actions["wvc_edit"] = sprintf(
                '<a href="%s" aria-label="%s">%s</a>',
                esc_url($custom_edit_url),
                esc_attr__("Edit with 10Web AI Builder", "wvc-theme"),
                esc_html__("Edit with WVC Editor", "wvc-theme")
            );
        }
        
        return $actions;
    }

    public function customize_edit_post_link($link, $post_id, $context) {
        if (!is_admin()) {
            return $link;
        }

        // Check if we're on the edit-page screen and dealing with a page
        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && $screen->id === 'edit-page' && get_post_type($post_id) === 'page') {
                
                // Keep capability check consistent with core
                if (current_user_can('edit_page', $post_id)) {
                    // Check if page is built with WVC editor
                    $post = get_post($post_id);
                    $is_built_with_wvc = $this->is_built_with_wvc($post);
                    
                    // If built with WVC, modify the link to use wvc-editor action
                    if ($is_built_with_wvc) {
                        // Handle encoded ampersands in display context
                        if ($context === 'display') {
                            // Replace the encoded action parameter
                            $link = preg_replace('/&amp;action=([^&]+)/', '', $link);
                            $link = add_query_arg("action", "wvc-editor", $link);
                        } else {
                            // For raw context, use remove_query_arg
                            $link = remove_query_arg("action", $link);
                            $link = add_query_arg("action", "wvc-editor", $link);
                        }
                    }
                }
            }
        }
        
        return $link;
    }
    

    
    /**
     * Check if post is built with WVC editor
     * 
     * @param \WP_Post $post Post object
     * @return bool
     */
    public function is_built_with_wvc($post) {
        if (!$post || !$post->ID) {
            return false;
        }
        
        return \WVC\Core\Core::is_built_with_wvc($post->ID);
    }
    
    /**
     * Check if current page is a WVC editor page
     * 
     * @return bool True if current page is WVC editor page
     */
    public static function is_editor_page()
    {
        return \WVC\Editor\Editor::is_editor_page_static();
    }

    /**
     * Add admin body classes for WVC editor status
     * 
     * @param string $classes Existing admin body classes
     * @return string Modified admin body classes
     */
    public function body_status_classes($classes) {
        // Case 1: We're on the WVC editor page itself
        if (self::is_editor_page()) {
            $classes .= ' wvc-editor-active';
            return $classes;
        }
        
        // Case 2: We're on a regular post.php page for a page
        global $pagenow;
        if ($pagenow === 'post.php' && isset($_GET['post'])) {
            $post = get_post(intval($_GET['post']));
            if ($post && $post->post_type === 'page') {
                $is_built_with_wvc = $this->is_built_with_wvc($post);
                
                if ($is_built_with_wvc) {
                    $classes .= ' wvc-editor-built';
                } else {
                    $classes .= ' wvc-editor-not-built';
                }
            }
        }
        
        return $classes;
    }

    
    /**
     * Modify title for WVC-built pages in admin
     * 
     * @param string $title The post title
     * @param int $post_id The post ID
     * @return string Modified title
     */
    public function modify_title_for_wvc_pages($title, $post_id) {
        // Only apply in admin and for pages
        if (!is_admin() || get_post_type($post_id) !== 'page') {
            return $title;
        }
        
        // Check if we're on the pages list screen
        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if (!$screen || $screen->id !== 'edit-page') {
                return $title;
            }
        }
        
        // Check if page is built with WVC
        $post = get_post($post_id);
        if ($this->is_built_with_wvc($post)) {
            $title .= ' — ' . __('WVC Editor', 'wvc-theme');
        }
        
        return $title;
    }
    

}
