<?php

namespace WVC\FormSubmissions;

/**
 * Admin Handlers - POST/AJAX Handlers
 *
 * Handle all POST and AJAX requests
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Handle form submission update
 */
function wvc_handle_update_form_submission() {
    // Verify nonce
    $submission_id = isset($_POST['submission_id']) ? intval($_POST['submission_id']) : 0;

    if (!isset($_POST['wvc_edit_nonce']) || !wp_verify_nonce($_POST['wvc_edit_nonce'], 'wvc_edit_submission_' . $submission_id)) {
        wp_die('Security check failed');
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';
    $fields = isset($_POST['fields']) ? $_POST['fields'] : array();

    if (!$submission_id || !$form_id) {
        wp_redirect(admin_url('admin.php?page=wvc_form_overview'));
        exit;
    }

    // Verify submission exists
    $submission = get_post($submission_id);
    if (!$submission || $submission->post_type !== 'wvc_form_data') {
        wp_die('Invalid submission');
    }

    // Get form fields from schema for proper sanitization
    $form_fields_specs = array();
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    if (!empty($forms)) {
        $form = $forms[0];
        $form_fields_specs = get_post_meta($form->ID, 'wvc_form_fields', true) ?: array();
    }

    // Sanitize and update individual field meta
    $updated_fields = array();
    foreach ($fields as $field_name => $field_value) {
        $sanitized_name = sanitize_key($field_name);

        // Get field spec from form schema
        $field_spec = null;
        foreach ($form_fields_specs as $spec) {
            $spec_key = $spec['key'] ?? $spec['name'] ?? '';
            if ($spec_key === $sanitized_name) {
                $field_spec = $spec;
                break;
            }
        }

        // Sanitize based on field type
        $field_type = $field_spec['type'] ?? 'text';
        $sanitized_value = wvc_sanitize_submission_field_value($field_value, $field_type);

        // Update individual field meta
        update_post_meta($submission_id, '_wvc_field_' . $sanitized_name, $sanitized_value);

        $updated_fields[$sanitized_name] = $sanitized_value;
    }

    // Update the JSON form data
    update_post_meta($submission_id, '_wvc_form_data', wp_json_encode($updated_fields));

    // Update the complete form data array
    update_post_meta($submission_id, '_wvc_form_data_complete', $updated_fields);

    // Update post title with first field value (for better admin display)
    if (!empty($updated_fields)) {
        $first_value = reset($updated_fields);
        wp_update_post(array(
            'ID' => $submission_id,
            'post_title' => 'Submission: ' . substr($first_value, 0, 50)
        ));
    }

    // Redirect back to submissions page with success message
    wp_redirect(admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id) . '&updated=1'));
    exit;
}

/**
 * Handle form submission deletion
 */
function wvc_handle_delete_form_submission() {
    // Verify nonce
    $submission_id = isset($_GET['submission_id']) ? intval($_GET['submission_id']) : 0;

    if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wvc_delete_submission_' . $submission_id)) {
        wp_die('Security check failed');
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    $form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';

    if ($submission_id) {
        // Force delete the submission (bypass trash)
        wp_delete_post($submission_id, true);
    }

    // Redirect back to submissions page with success message
    wp_redirect(admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id) . '&deleted=1'));
    exit;
}

/**
 * Handle email notification settings save
 */
function wvc_handle_save_email_settings() {
    // Get form ID
    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';

    if (empty($form_id)) {
        wp_die('Invalid form ID');
    }

    // Verify nonce
    if (!isset($_POST['wvc_email_settings_nonce']) || !wp_verify_nonce($_POST['wvc_email_settings_nonce'], 'wvc_email_settings_' . $form_id)) {
        wp_die('Security check failed');
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    // Get the wvc_form post
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    if (empty($forms)) {
        wp_die('Form not found');
    }

    $form_post = $forms[0];

    // Get and sanitize inputs
    $email_enabled = isset($_POST['email_enabled']) ? true : false;
    $email_addresses_raw = isset($_POST['email_addresses']) && is_array($_POST['email_addresses']) ? $_POST['email_addresses'] : array();

    // Validate email addresses
    $email_addresses = array();
    $invalid_emails = array();

    if (!empty($email_addresses_raw)) {
        foreach ($email_addresses_raw as $email) {
            $email = sanitize_email(trim($email));

            if (empty($email)) {
                continue;
            }

            if (is_email($email)) {
                $email_addresses[] = $email;
            } else {
                $invalid_emails[] = $email;
            }
        }
    }

    // Remove duplicates
    $email_addresses = array_unique($email_addresses);

    // Prepare settings array
    $settings = array(
        'enabled' => $email_enabled,
        'emails'  => $email_addresses
    );

    // Save to post meta
    update_post_meta($form_post->ID, 'wvc_form_email_settings', $settings);

    // Log the save operation
    error_log(sprintf(
        'WVC Email Settings Saved - Form: %s, Enabled: %s, Emails: %s',
        $form_id,
        $email_enabled ? 'Yes' : 'No',
        implode(', ', $email_addresses)
    ));

    // Prepare redirect URL with status message
    $redirect_url = admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id));

    if (!empty($invalid_emails)) {
        // Partial success - some emails were invalid
        $redirect_url .= '&settings_saved=partial&invalid=' . urlencode(implode(', ', $invalid_emails));
    } else {
        // Full success
        $redirect_url .= '&settings_saved=success';
    }

    wp_redirect($redirect_url);
    exit;
}

/**
 * Handle test email AJAX request
 */
function wvc_handle_test_email() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wvc_test_email_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }

    // Check permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Unauthorized'));
    }

    // Get parameters
    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';
    $emails = isset($_POST['emails']) && is_array($_POST['emails']) ? $_POST['emails'] : array();

    if (empty($form_id) || empty($emails)) {
        wp_send_json_error(array('message' => 'Missing required parameters'));
    }

    // Validate emails
    $valid_emails = array();
    foreach ($emails as $email) {
        $email = sanitize_email(trim($email));
        if (is_email($email)) {
            $valid_emails[] = $email;
        }
    }

    if (empty($valid_emails)) {
        wp_send_json_error(array('message' => 'No valid email addresses provided'));
    }

    // Get form label
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    $form_label = !empty($forms) ? $forms[0]->post_title : $form_id;

    // Create test submission data
    $test_submission_data = array(
        'post_id' => 0,
        'form_id' => $form_id,
        'section_name' => 'Test',
        'timestamp' => time() * 1000,
        'session_id' => 'test-session',
        'page_url' => admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id)),
        'user_agent' => 'WordPress Admin Test',
        'submission_attempt' => 1,
        'form_version' => '1.0.0',
        'validation_errors' => '',
        'individual_fields' => array(
            'test_field_1' => 'This is a test submission',
            'test_field_2' => 'Testing email notification functionality',
            'test_email' => $valid_emails[0],
        ),
        'submission_metadata' => array(
            'user_ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'submission_source' => 'admin_test',
        ),
        'submission_date' => date('Y-m-d H:i:s'),
    );

    // Generate email content
    $html_content = wvc_generate_submission_email_html($test_submission_data, $form_label . ' (TEST)');

    // Prepare email
    $subject = sprintf('[TEST] [%s] New Form Submission - %s',
        get_bloginfo('name'),
        $form_label
    );

    // Set email headers for HTML
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>',
    );

    // Send to each recipient
    $success_count = 0;
    $fail_count = 0;

    // Log email details before sending
    error_log("=== WVC Test Email Debug ===");
    error_log("Form ID: {$form_id}");
    error_log("Form Label: {$form_label}");
    error_log("Subject: {$subject}");
    error_log("Recipients: " . implode(', ', $valid_emails));
    error_log("From: " . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
    error_log("Headers: " . print_r($headers, true));
    error_log("Content Length: " . strlen($html_content) . " bytes");
    error_log("Content Preview: " . substr(strip_tags($html_content), 0, 200) . "...");

    // Check server mail configuration
    error_log("PHP mail() available: " . (function_exists('mail') ? 'YES' : 'NO'));
    error_log("Sendmail path: " . ini_get('sendmail_path'));
    error_log("SMTP: " . ini_get('SMTP'));
    error_log("smtp_port: " . ini_get('smtp_port'));

    // Hook to capture PHPMailer errors
    $phpmailer_error = '';
    $phpmailer_init_called = false;

    add_action('wp_mail_failed', function($wp_error) use (&$phpmailer_error) {
        $phpmailer_error = $wp_error->get_error_message();
        error_log("WVC Test Email - PHPMailer Error: " . $phpmailer_error);
        error_log("WVC Test Email - Error Data: " . print_r($wp_error->get_error_data(), true));
    });

    add_action('phpmailer_init', function($phpmailer) use (&$phpmailer_init_called) {
        $phpmailer_init_called = true;
        error_log("WVC Test Email - PHPMailer Init Called");
        error_log("WVC Test Email - Mailer Type: " . $phpmailer->Mailer);
        error_log("WVC Test Email - Host: " . $phpmailer->Host);
        error_log("WVC Test Email - Port: " . $phpmailer->Port);
        error_log("WVC Test Email - From: " . $phpmailer->From);
        error_log("WVC Test Email - FromName: " . $phpmailer->FromName);
    });

    foreach ($valid_emails as $email) {
        error_log("--- Attempting to send to: {$email} ---");

        $sent = wp_mail($email, $subject, $html_content, $headers);

        error_log("wp_mail() returned: " . ($sent ? 'TRUE' : 'FALSE'));
        error_log("phpmailer_init hook called: " . ($phpmailer_init_called ? 'YES' : 'NO'));

        if ($sent) {
            $success_count++;
            error_log("WVC Test Email: wp_mail() returned TRUE for {$email}");
            error_log("WVC Test Email: Email queued/sent to {$email}");
            error_log("NOTE: wp_mail() returning TRUE only means the email was handed to the mail system.");
            error_log("It does NOT guarantee delivery. Check spam folder or configure SMTP plugin.");
        } else {
            $fail_count++;
            error_log("WVC Test Email: wp_mail() returned FALSE for {$email}");
            error_log("WVC Test Email: Failed to queue/send to {$email}");
            if (!empty($phpmailer_error)) {
                error_log("WVC Test Email: PHPMailer Error Details: {$phpmailer_error}");
            }
        }
    }

    error_log("=== End Test Email Debug ===");

    // Log summary
    error_log(sprintf(
        'WVC Test Email Summary - Form: %s, Sent: %d, Failed: %d, Recipients: %s',
        $form_id,
        $success_count,
        $fail_count,
        implode(', ', $valid_emails)
    ));

    // Return response
    if ($success_count > 0) {
        $logs_url = wvc_get_tenweb_logs_url();

        if ($logs_url) {
            $message = sprintf(
                'Test email queued to %d recipient%s (%s). <a href="%s" target="_blank">View logs in 10Web Dashboard</a> for delivery details.',
                $success_count,
                $success_count > 1 ? 's' : '',
                implode(', ', $valid_emails),
                esc_url($logs_url)
            );

            $note = sprintf(
                'On local development (MAMP), emails may not be delivered. <a href="%s" target="_blank">Check 10Web logs</a> for detailed information.',
                esc_url($logs_url)
            );
        } else {
            // Fallback if 10Web dashboard not available
            $message = sprintf(
                'Test email queued to %d recipient%s (%s).',
                $success_count,
                $success_count > 1 ? 's' : '',
                implode(', ', $valid_emails)
            );

            $note = 'On local development (MAMP), emails may not be delivered. Check your server logs for detailed information.';
        }

        wp_send_json_success(array(
            'message' => $message,
            'logs_url' => $logs_url,
            'note' => $note
        ));
    } else {
        $logs_url = wvc_get_tenweb_logs_url();

        if ($logs_url) {
            $error_message = sprintf(
                'Failed to send test email. <a href="%s" target="_blank">Check 10Web logs</a> for error details.',
                esc_url($logs_url)
            );
        } else {
            $error_message = 'Failed to send test email. Check your server logs for error details.';
        }

        wp_send_json_error(array(
            'message' => $error_message
        ));
    }
}

/**
 * Handle hook settings save
 */
function wvc_handle_save_hook_settings() {
    // Get form ID
    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';

    if (empty($form_id)) {
        wp_die('Invalid form ID');
    }

    // Verify nonce
    if (!isset($_POST['wvc_hook_settings_nonce']) || !wp_verify_nonce($_POST['wvc_hook_settings_nonce'], 'wvc_hook_settings_' . $form_id)) {
        wp_die('Security check failed');
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    // Get the wvc_form post
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    if (empty($forms)) {
        wp_die('Form not found');
    }

    $form_post = $forms[0];

    // Get and sanitize inputs
    $hooks_enabled = isset($_POST['hooks_enabled']) ? true : false;
    $hooks_raw = isset($_POST['hooks']) && is_array($_POST['hooks']) ? $_POST['hooks'] : array();

    // Validate and sanitize hooks
    $hooks = array();
    $invalid_hooks = array();

    foreach ($hooks_raw as $hook) {
        // Skip if missing required fields
        if (empty($hook['id']) || empty($hook['type'])) {
            continue;
        }

        $hook_id = sanitize_text_field($hook['id']);
        $hook_type = sanitize_text_field($hook['type']);
        $hook_enabled = isset($hook['enabled']) ? true : false;
        $hook_name = isset($hook['name']) ? sanitize_text_field($hook['name']) : '';
        $webhook_url = isset($hook['webhook_url']) ? esc_url_raw($hook['webhook_url'], array('https')) : '';
        $channel = isset($hook['channel']) ? sanitize_text_field($hook['channel']) : '';

        // Validate webhook URL (must be HTTPS)
        if (!empty($webhook_url)) {
            if (!filter_var($webhook_url, FILTER_VALIDATE_URL)) {
                $invalid_hooks[] = array(
                    'name' => $hook_name,
                    'error' => 'Invalid URL format'
                );
                continue;
            }

            if (strpos($webhook_url, 'https://') !== 0) {
                $invalid_hooks[] = array(
                    'name' => $hook_name,
                    'error' => 'URL must use HTTPS'
                );
                continue;
            }
        } else {
            // Skip hooks without webhook URL
            continue;
        }

        // Add to valid hooks
        $hooks[] = array(
            'id' => $hook_id,
            'type' => $hook_type,
            'enabled' => $hook_enabled,
            'name' => $hook_name,
            'webhook_url' => $webhook_url,
            'channel' => $channel,
        );
    }

    // Prepare settings array
    $settings = array(
        'enabled' => $hooks_enabled,
        'hooks'   => $hooks
    );

    // Save to post meta
    update_post_meta($form_post->ID, 'wvc_form_hook_settings', $settings);

    // Log the save operation
    error_log(sprintf(
        'WVC Hook Settings Saved - Form: %s, Enabled: %s, Hooks Count: %d',
        $form_id,
        $hooks_enabled ? 'Yes' : 'No',
        count($hooks)
    ));

    // Prepare redirect URL with status message
    $redirect_url = admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id));

    if (!empty($invalid_hooks)) {
        // Partial success - some hooks were invalid
        $error_messages = array();
        foreach ($invalid_hooks as $invalid) {
            $error_messages[] = $invalid['name'] . ': ' . $invalid['error'];
        }
        $redirect_url .= '&hook_settings_saved=partial&invalid=' . urlencode(implode('; ', $error_messages));
    } else {
        // Full success
        $redirect_url .= '&hook_settings_saved=success';
    }

    wp_redirect($redirect_url);
    exit;
}

/**
 * Handle test hook AJAX request
 */
function wvc_handle_test_hook() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wvc_test_hook_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }

    // Check permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Unauthorized'));
    }

    // Get parameters
    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';
    $hook_config = isset($_POST['hook']) && is_array($_POST['hook']) ? $_POST['hook'] : array();

    if (empty($form_id) || empty($hook_config)) {
        wp_send_json_error(array('message' => 'Missing required parameters'));
    }

    // Sanitize hook config
    $hook = array(
        'type' => isset($hook_config['type']) ? sanitize_text_field($hook_config['type']) : 'slack',
        'name' => isset($hook_config['name']) ? sanitize_text_field($hook_config['name']) : 'Test Hook',
        'webhook_url' => isset($hook_config['webhook_url']) ? esc_url_raw($hook_config['webhook_url'], array('https')) : '',
        'channel' => isset($hook_config['channel']) ? sanitize_text_field($hook_config['channel']) : '',
    );

    // Validate webhook URL
    if (empty($hook['webhook_url']) || !filter_var($hook['webhook_url'], FILTER_VALIDATE_URL)) {
        wp_send_json_error(array('message' => 'Invalid webhook URL'));
    }

    if (strpos($hook['webhook_url'], 'https://') !== 0) {
        wp_send_json_error(array('message' => 'Webhook URL must use HTTPS'));
    }

    // Get form label
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));

    $form_label = !empty($forms) ? $forms[0]->post_title : $form_id;

    // Create test submission data
    $test_submission_data = array(
        'post_id' => 0,
        'form_id' => $form_id,
        'section_name' => 'Test',
        'timestamp' => time() * 1000,
        'session_id' => 'test-session',
        'page_url' => admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id)),
        'user_agent' => 'WordPress Admin Test',
        'submission_attempt' => 1,
        'form_version' => '1.0.0',
        'validation_errors' => '',
        'individual_fields' => array(
            'test_field_1' => 'This is a test submission',
            'test_field_2' => 'Testing webhook notification functionality',
            'test_field_3' => 'Sample data for testing purposes',
        ),
        'submission_metadata' => array(
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'submission_source' => 'admin_test',
        ),
        'submission_date' => date('Y-m-d H:i:s'),
    );

    // Log test hook attempt
    error_log(sprintf(
        'WVC Test Hook - Form: %s, Type: %s, Hook Name: %s, URL: %s',
        $form_id,
        $hook['type'],
        $hook['name'],
        $hook['webhook_url']
    ));

    // Send test hook based on type
    $result = false;
    if ($hook['type'] === 'slack') {
        $result = wvc_send_slack_notification(0, $form_id, $hook);
    }

    // Return response
    if ($result) {
        wp_send_json_success(array(
            'message' => sprintf('Test message sent successfully to %s!', $hook['name'])
        ));
    } else {
        $logs_url = wvc_get_tenweb_logs_url();

        if ($logs_url) {
            $error_message = sprintf(
                'Failed to send test message. <a href="%s" target="_blank">Check 10Web logs</a> for error details.',
                esc_url($logs_url)
            );
        } else {
            $error_message = 'Failed to send test message. Check your server logs for error details.';
        }

        wp_send_json_error(array(
            'message' => $error_message
        ));
    }
}

/**
 * Handle CSV export of form submissions
 */
function wvc_handle_export_submissions_csv() {
    // Get form ID
    $form_id = isset($_POST['form_id']) ? sanitize_text_field($_POST['form_id']) : '';

    if (empty($form_id)) {
        wp_die('Invalid form ID');
    }

    // Verify nonce
    $nonce_action = 'wvc_export_csv_' . $form_id;
    if (!isset($_POST['wvc_export_nonce']) || !wp_verify_nonce($_POST['wvc_export_nonce'], $nonce_action)) {
        wp_die('Security check failed');
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }

    // Get filter parameters
    $filters = array(
        'date_range' => isset($_POST['date_range']) ? sanitize_text_field($_POST['date_range']) : 'all',
        'date_from' => isset($_POST['date_from']) ? sanitize_text_field($_POST['date_from']) : '',
        'date_to' => isset($_POST['date_to']) ? sanitize_text_field($_POST['date_to']) : '',
    );

    // Get submissions
    $submissions = wvc_get_submissions_for_export($form_id, $filters);

    if (empty($submissions)) {
        // Redirect back with error message
        $redirect_url = add_query_arg(
            array(
                'page' => 'wvc_form_submissions',
                'form_id' => $form_id,
                'export_error' => 'no_submissions'
            ),
            admin_url('admin.php')
        );
        wp_redirect($redirect_url);
        exit;
    }

    // Get form fields from first submission or schema
    $form_fields = wvc_get_form_fields_by_form_key($form_id);

    // Prepare CSV filename
    $datetime = date('Y-m-d_H-i-s');
    $filename = sanitize_file_name($form_id . '-' . $datetime . '.csv');

    // Set headers for download
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');

    // Open output stream
    $output = fopen('php://output', 'w');

    // Write CSV header row
    $header = array(
        'Submission ID',
        'Submission Date',
    );

    // Add form field columns
    foreach ($form_fields as $field) {
        $header[] = ucwords(str_replace('_', ' ', $field));
    }

    // Add metadata columns
    $header[] = 'Form Version';
    $header[] = 'User IP';
    $header[] = 'User Agent';

    fputcsv($output, $header);

    // Write data rows
    foreach ($submissions as $submission) {
        $row = array(
            $submission['post_id'],
            $submission['submission_date'],
        );

        // Add form field values
        foreach ($form_fields as $field) {
            $value = '';
            if (isset($submission['individual_fields'][$field])) {
                $value = $submission['individual_fields'][$field];
            } elseif (isset($submission['individual_fields_lowercase_map'][strtolower($field)])) {
                $value = $submission['individual_fields_lowercase_map'][strtolower($field)];
            }

            // Sanitize for CSV (prevent formula injection)
            if (is_string($value) && strlen($value) > 0 && in_array($value[0], array('=', '+', '-', '@'))) {
                $value = "'" . $value;
            }

            $row[] = $value;
        }

        // Add metadata
        $row[] = !empty($submission['form_version']) ? 'v' . $submission['form_version'] : '';

        // Get user_ip from submission_metadata
        $user_ip = '';
        if (!empty($submission['submission_metadata']['user_ip'])) {
            $user_ip = $submission['submission_metadata']['user_ip'];
        }
        $row[] = $user_ip;

        $row[] = !empty($submission['user_agent']) ? $submission['user_agent'] : '';

        fputcsv($output, $row);
    }

    fclose($output);
    exit;
}
