<?php

namespace WVC\FormSubmissions;

/**
 * Admin Pages - Page Rendering Functions
 *
 * Render admin pages (form overview, submissions table, edit submission)
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit;
}

/**
 * Display form overview page with all registered forms and their submission counts
 */
function wvc_form_overview_page() {
    // Get all registered forms with counts (includes forms with 0 submissions)
    $form_stats = wvc_get_all_registered_forms_with_counts();
    $total_submissions = wvc_get_total_submissions_count();

    echo '<div class="wrap">';
    echo '<h1>Form Overview</h1>';

    // Add warning banner about publishing requirement
    // echo '<div class="notice notice-warning" style="margin: 15px 0; padding: 12px 15px; border-left: 4px solid #ffb900; background-color: #fff8e5;">';
    // echo '</div>';

    if (empty($form_stats)) {
        echo '<div class="notice notice-info"><p>No forms registered yet. Forms will appear here after they are created via the website generation process.</p></div>';
        echo '</div>';
        return;
    }

    // Summary statistics
    echo '<div class="wvc-stats-summary" style="padding: 20px; margin: 20px 0; border-radius: 5px;">';
    echo '<h3>Summary</h3>';
    echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">';
    echo '<div><strong>Total Forms:</strong> ' . count($form_stats) . '</div>';
    echo '<div><strong>Total Submissions:</strong> ' . $total_submissions . '</div>';

    // Count forms with submissions
    $forms_with_submissions = 0;
    foreach ($form_stats as $stat) {
        if ($stat['submission_count'] > 0) {
            $forms_with_submissions++;
        }
    }
    echo '<div><strong>Forms with Submissions:</strong> ' . $forms_with_submissions . '</div>';
    echo '<div><strong>Forms without Submissions:</strong> ' . (count($form_stats) - $forms_with_submissions) . '</div>';
    echo '</div>';
    echo '</div>';

    // Detailed table view
    echo '<div style="margin-top: 30px;">';
    echo '<h3>All Forms</h3>';
    echo '<div class="wvc-forms-table">';
    echo '<table class="wp-list-table widefat fixed striped">';
    echo '<thead>';
    echo '<tr>';
    echo '<th scope="col" style="width: 22%;">Form Name</th>';
    echo '<th scope="col" style="width: 16%;">Form Key</th>';
    echo '<th scope="col" style="width: 7%;">Version</th>';
    echo '<th scope="col" style="width: 9%;">Submissions</th>';
    echo '<th scope="col" style="width: 11%;">Email Status</th>';
    echo '<th scope="col" style="width: 11%;">Hook Status</th>';
    echo '<th scope="col" style="width: 9%;">Status</th>';
    echo '<th scope="col" style="width: 15%;">Actions</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';

    foreach ($form_stats as $form_stat) {
        $form_key = esc_html($form_stat['form_key']);
        $form_label = esc_html($form_stat['form_label']);
        $submission_count = $form_stat['submission_count'];

        // Generate URLs
        $view_submissions_url = admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_stat['form_key']));

        // Get email settings to show status
        $email_settings = wvc_get_email_notification_settings($form_stat['form_key']);
        $email_enabled = !empty($email_settings['enabled']);
        $email_count = !empty($email_settings['emails']) ? count($email_settings['emails']) : 0;

        // Email status display
        if ($email_enabled && $email_count > 0) {
            $email_status = '<span style="color: #46b450; font-weight: 500;">✓ Enabled (' . $email_count . ')</span>';
        } elseif ($email_count > 0) {
            $email_status = '<span style="color: #ffc107; font-weight: 500;">⚠ Disabled (' . $email_count . ')</span>';
        } else {
            $email_status = '<span style="color: #999;">Not configured</span>';
        }

        // Get hook settings to show status
        $hook_settings = wvc_get_hook_settings($form_stat['form_key']);
        $hooks_enabled = !empty($hook_settings['enabled']);
        $hook_count = !empty($hook_settings['hooks']) ? count($hook_settings['hooks']) : 0;

        // Hook status display
        if ($hooks_enabled && $hook_count > 0) {
            $hook_status = '<span style="color: #28a745; font-weight: 500;">✓ Enabled (' . $hook_count . ')</span>';
        } elseif ($hook_count > 0) {
            $hook_status = '<span style="color: #ffc107; font-weight: 500;">⚠ Disabled (' . $hook_count . ')</span>';
        } else {
            $hook_status = '<span style="color: #999;">Not configured</span>';
        }

        // Determine status
        $status = $submission_count > 0 ? 'Active' : 'Registered';
        $status_color = $submission_count > 0 ? '#46b450' : '#999';

        // Get form version
        $form_version = isset($form_stat['form_version']) ? $form_stat['form_version'] : null;
        $version_display = !empty($form_version) ? 'v' . $form_version : '<span style="color: #999;">—</span>';

        echo '<tr>';
        echo '<td><strong>' . $form_label . '</strong></td>';
        echo '<td><code>' . $form_key . '</code></td>';
        echo '<td>' . $version_display . '</td>';
        echo '<td>' . $submission_count . '</td>';
        echo '<td>' . $email_status . '</td>';
        echo '<td>' . $hook_status . '</td>';
        echo '<td><span style="color: ' . $status_color . '; font-weight: 500;">● ' . $status . '</span></td>';
        echo '<td style="white-space: nowrap;">';

        // Always show "Manage" button to access submissions & settings
        echo '<a href="' . esc_url($view_submissions_url) . '" class="button button-primary" title="View submissions and configure email notifications">Manage</a>';

        echo '</td>';
        echo '</tr>';
    }

    echo '</tbody>';
    echo '</table>';
    echo '</div>';
    echo '</div>';

    echo '</div>';
}

/**
 * Display form-specific submissions page with dynamic columns
 */
function wvc_form_submissions_page() {
    // Get form ID from URL parameter
    $form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';

    if (empty($form_id)) {
        echo '<div class="wrap">';
        echo '<h1>Form Submissions</h1>';
        echo '<div class="notice notice-error"><p>No form ID specified. Please select a form from the <a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">Form Overview</a> page.</p></div>';
        echo '</div>';
        return;
    }

    // Get form fields from registered schema or first submission
    $form_fields = wvc_get_form_fields_by_form_key($form_id);

    // Get form field specs for proper labels
    $form_fields_specs = array();
    $forms = get_posts(array(
        'post_type'   => 'wvc_form',
        'meta_key'    => 'wvc_form_key',
        'meta_value'  => $form_id,
        'post_status' => 'publish',
        'numberposts' => 1,
    ));
    if (!empty($forms)) {
        $form = $forms[0];
        $form_fields_specs = get_post_meta($form->ID, 'wvc_form_fields', true) ?: array();
    }

    // Create field label map
    $field_labels = array();
    foreach ($form_fields_specs as $field_spec) {
        $field_key = $field_spec['key'] ?? $field_spec['name'] ?? '';
        $field_label = $field_spec['label'] ?? ucwords(str_replace('_', ' ', $field_key));
        if (!empty($field_key)) {
            $field_labels[$field_key] = $field_label;
        }
    }

    // Get all submissions for this form
    $submissions = wvc_get_form_submissions_with_fields($form_id);

    // Get submission count
    $submission_count = wvc_get_form_submission_count($form_id);

    echo '<div class="wrap">';

    // Show success/error messages
    if (isset($_GET['updated']) && $_GET['updated'] == '1') {
        echo '<div class="notice notice-success is-dismissible"><p>Submission updated successfully.</p></div>';
    }
    if (isset($_GET['deleted']) && $_GET['deleted'] == '1') {
        echo '<div class="notice notice-success is-dismissible"><p>Submission deleted successfully.</p></div>';
    }
    if (isset($_GET['settings_saved'])) {
        if ($_GET['settings_saved'] === 'success') {
            echo '<div class="notice notice-success is-dismissible"><p>Email notification settings saved successfully.</p></div>';
        } elseif ($_GET['settings_saved'] === 'partial') {
            $invalid = isset($_GET['invalid']) ? sanitize_text_field($_GET['invalid']) : '';
            echo '<div class="notice notice-warning is-dismissible"><p>Settings saved, but some email addresses were invalid: ' . esc_html($invalid) . '</p></div>';
        }
    }

    echo '<h1>Form Submissions: ' . esc_html($form_id) . '</h1>';

    // Add navigation breadcrumb
    echo '<p class="wvc-breadcrumb">';
    echo '<a href="' . admin_url('edit.php?post_type=wvc_form_data&page=wvc_form_overview') . '">← Back to Form Overview</a>';
    // echo ' | <a href="' . admin_url('edit.php?post_type=wvc_form_data') . '">All Submissions</a>';
    echo '</p>';

    // Show helpful notice if email not configured
    $temp_email_settings = wvc_get_email_notification_settings($form_id);
    if (empty($temp_email_settings['emails'])) {
        echo '<div class="notice notice-info" style="margin: 15px 0;">';
        echo '<p><strong>💡 Tip:</strong> Configure email notifications below to receive alerts when users submit this form.</p>';
        echo '</div>';
    }

    // Email notification settings panel
    $email_settings = wvc_get_email_notification_settings($form_id);
    $settings_enabled = $email_settings['enabled'];
    $settings_emails = !empty($email_settings['emails']) ? $email_settings['emails'] : array();
    $email_count = count($settings_emails);

    echo '<div class="wvc-form-settings">';

    // Collapsible header
    echo '<div class="wvc-settings-header">';
    echo '<h3>⚙️ Email Notification Settings';
    if ($email_count > 0) {
        echo '<span class="wvc-email-count" style="display: inline-block;">' . $email_count . '</span>';
    } else {
        echo '<span class="wvc-email-count" style="display: none;">0</span>';
    }
    echo '</h3>';
    echo '<span class="wvc-settings-toggle collapsed">▼</span>';
    echo '</div>';

    // Settings body (always collapsed by default)
    echo '<div class="wvc-settings-body">';

    // Status indicator
    $status_class = $settings_enabled ? 'enabled' : 'disabled';
    $status_icon = $settings_enabled ? '✅' : '⚠️';
    $status_text = $settings_enabled ? 'Email notifications are <strong>enabled</strong>' : 'Email notifications are <strong>disabled</strong>';

    echo '<div class="wvc-settings-status ' . $status_class . '">';
    echo '<span class="wvc-settings-status-icon">' . $status_icon . '</span>';
    echo '<span>' . $status_text . '</span>';
    echo '</div>';

    echo '<form method="post" action="' . admin_url('admin-post.php') . '">';
    echo '<input type="hidden" name="action" value="wvc_save_email_settings" />';
    echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '" />';
    echo wp_nonce_field('wvc_email_settings_' . $form_id, 'wvc_email_settings_nonce', true, false);

    echo '<table class="form-table" role="presentation">';

    // Enable/disable checkbox
    echo '<tr>';
    echo '<th scope="row"><label for="wvc_email_enabled">Enable Notifications</label></th>';
    echo '<td>';
    echo '<label for="wvc_email_enabled">';
    echo '<input type="checkbox" id="wvc_email_enabled" name="email_enabled" value="1" ' . checked($settings_enabled, true, false) . ' />';
    echo ' Send email notifications when this form is submitted';
    echo '</label>';
    echo '</td>';
    echo '</tr>';

    // Email addresses - dynamic list
    echo '<tr>';
    echo '<th scope="row"><label for="wvc_new_email">Recipient Emails</label></th>';
    echo '<td>';

    // Add email input group
    echo '<div class="wvc-email-input-group">';
    echo '<input type="email" id="wvc_new_email" class="regular-text" placeholder="email@example.com" />';
    echo '<button type="button" id="wvc_add_email_btn" class="button button-secondary">Add Email</button>';
    echo '</div>';

    // Email list
    echo '<ul id="wvc_email_list" class="wvc-email-list">';
    foreach ($settings_emails as $email) {
        echo '<li class="wvc-email-list-item" data-email="' . esc_attr($email) . '">';
        echo '<span class="wvc-email-list-item-text">' . esc_html($email) . '</span>';
        echo '<input type="hidden" name="email_addresses[]" value="' . esc_attr($email) . '" />';
        echo '<a href="#" class="button-link-delete wvc-remove-email">Remove</a>';
        echo '</li>';
    }
    echo '</ul>';

    echo '<p class="description">Add email addresses to receive notifications when the form is submitted.</p>';
    echo '</td>';
    echo '</tr>';

    // // Email configuration diagnostics
    // echo '<tr>';
    // echo '<th scope="row">Mail Configuration</th>';
    // echo '<td>';
    // echo '<div style="background: #f8f9fa; padding: 10px; border-radius: 4px; font-family: monospace; font-size: 12px;">';
    // echo '<strong>Server Mail Status:</strong><br>';
    // echo 'PHP mail() function: ' . (function_exists('mail') ? '<span style="color: #28a745;">✓ Available</span>' : '<span style="color: #dc3545;">✗ Not Available</span>') . '<br>';
    // echo 'Sendmail path: <code>' . esc_html(ini_get('sendmail_path')) . '</code><br>';
    // echo 'Admin email: <code>' . esc_html(get_option('admin_email')) . '</code><br>';
    // echo '<br><strong>Note:</strong> MAMP/Local development may not send real emails. <br>';
    // echo 'For production, install <strong>WP Mail SMTP</strong> plugin or configure your mail server.<br>';
    // echo 'Click "Send Test Email" to verify email delivery.';
    // echo '</div>';
    // echo '</td>';
    // echo '</tr>';

    echo '</table>';

    // Submit button
    echo '<div class="submit" style="display: flex; gap: 10px; align-items: center;">';
    echo '<input type="submit" name="submit" class="button button-primary" value="Save Email Settings" />';
    echo '<button type="button" id="wvc_test_email_btn" class="button button-secondary">Send Test Email</button>';
    echo '<span id="wvc_test_email_result" style="margin-left: 10px;"></span>';
    echo '</div>';

    echo '</form>';
    echo '</div>'; // .wvc-settings-body
    echo '</div>'; // .wvc-form-settings

    // Export submissions panel
    $export_nonce_action = 'wvc_export_csv_' . $form_id;
    $export_row_count = min($submission_count, 1000);

    echo '<div class="wvc-form-settings wvc-export-panel">';

    // Collapsible header
    echo '<div class="wvc-settings-header">';
    echo '<h3>📥 Export Submissions</h3>';
    echo '<span class="wvc-settings-toggle collapsed">▼</span>';
    echo '</div>';

    // Settings body (collapsed by default)
    echo '<div class="wvc-settings-body">';

    echo '<form method="post" action="' . admin_url('admin-post.php') . '">';
    echo '<input type="hidden" name="action" value="wvc_export_submissions_csv" />';
    echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '" />';
    echo wp_nonce_field($export_nonce_action, 'wvc_export_nonce', true, false);

    echo '<table class="form-table" role="presentation">';

    // Date range selector
    echo '<tr>';
    echo '<th scope="row"><label for="wvc_export_date_range">Date Range</label></th>';
    echo '<td>';
    echo '<select name="date_range" id="wvc_export_date_range" class="regular-text">';
    echo '<option value="all">All Time</option>';
    echo '<option value="today">Today</option>';
    echo '<option value="last_7">Last 7 Days</option>';
    echo '<option value="last_30">Last 30 Days</option>';
    echo '<option value="last_90">Last 90 Days</option>';
    echo '<option value="custom">Custom Range...</option>';
    echo '</select>';
    echo '</td>';
    echo '</tr>';

    // Custom date range inputs (hidden by default)
    echo '<tr id="wvc_custom_date_row" style="display: none;">';
    echo '<th scope="row">Custom Range</th>';
    echo '<td>';
    echo 'From: <input type="date" name="date_from" id="wvc_date_from" /> ';
    echo 'To: <input type="date" name="date_to" id="wvc_date_to" />';
    echo '</td>';
    echo '</tr>';

    // Row limit info
    echo '<tr>';
    echo '<th scope="row">Export Limit</th>';
    echo '<td>';
    echo '<p class="description">Maximum ' . $export_row_count . ' row' . ($export_row_count !== 1 ? 's' : '') . ' will be exported (limited to 1000).</p>';
    echo '</td>';
    echo '</tr>';

    echo '</table>';

    // Submit button
    echo '<div class="submit">';
    echo '<button type="submit" class="button button-primary">📥 Export CSV</button>';
    echo '</div>';

    echo '</form>';
    echo '</div>'; // .wvc-settings-body
    echo '</div>'; // .wvc-form-settings

    // Hooks Configuration Panel
    $hook_settings = wvc_get_hook_settings($form_id);
    $hooks_enabled = !empty($hook_settings['enabled']);
    $configured_hooks = !empty($hook_settings['hooks']) ? $hook_settings['hooks'] : array();

    echo '<div class="wvc-form-settings wvc-hooks-panel">';

    // Collapsible header
    echo '<div class="wvc-settings-header">';
    echo '<h3>🔗 Hooks Configuration</h3>';
    echo '<span class="wvc-settings-toggle collapsed">▼</span>';
    echo '</div>';

    // Settings body (collapsed by default)
    echo '<div class="wvc-settings-body">';

    echo '<form method="post" action="' . admin_url('admin-post.php') . '" id="wvc-hooks-settings-form">';
    echo '<input type="hidden" name="action" value="wvc_save_hook_settings" />';
    echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '" />';
    echo wp_nonce_field('wvc_hook_settings_' . $form_id, 'wvc_hook_settings_nonce', true, false);

    // Global enable/disable
    echo '<div class="wvc-hooks-global-toggle">';
    echo '<label>';
    echo '<input type="checkbox" name="hooks_enabled" value="1" ' . checked($hooks_enabled, true, false) . '>';
    echo ' Enable Hooks (notifications will be sent when forms are submitted)';
    echo '</label>';
    echo '</div>';

    // Hooks container
    echo '<div id="wvc-hooks-container">';

    // Render existing hooks
    if (!empty($configured_hooks)) {
        foreach ($configured_hooks as $hook) {
            $hook_id = esc_attr($hook['id'] ?? 'hook_' . uniqid());
            $hook_type = esc_attr($hook['type'] ?? 'slack');
            $hook_enabled = !empty($hook['enabled']);
            $hook_name = esc_attr($hook['name'] ?? '');
            $webhook_url = esc_url($hook['webhook_url'] ?? '');
            $channel = esc_attr($hook['channel'] ?? '');

            echo '<div class="wvc-hook-item" data-hook-id="' . $hook_id . '">';

            // Header
            echo '<div class="wvc-hook-item-header">';
            echo '<div class="wvc-hook-item-left">';
            echo '<span class="wvc-hook-type-badge">Slack</span>';
            echo '<label style="margin: 0;">';
            echo '<input type="checkbox" name="hooks[' . $hook_id . '][enabled]" value="1" ' . checked($hook_enabled, true, false) . '>';
            echo ' Enable this hook';
            echo '</label>';
            echo '</div>';
            echo '<div class="wvc-hook-item-right">';
            echo '<button type="button" class="wvc-hook-test-btn" data-hook-id="' . $hook_id . '">Test</button>';
            echo '<button type="button" class="wvc-hook-remove-btn" data-hook-id="' . $hook_id . '">Remove</button>';
            echo '</div>';
            echo '</div>';

            // Hidden fields
            echo '<input type="hidden" name="hooks[' . $hook_id . '][id]" value="' . $hook_id . '">';
            echo '<input type="hidden" name="hooks[' . $hook_id . '][type]" value="' . $hook_type . '">';

            // Hook name
            echo '<div class="wvc-hook-field">';
            echo '<label>Hook Name</label>';
            echo '<input type="text" name="hooks[' . $hook_id . '][name]" value="' . $hook_name . '" placeholder="e.g., Sales Team Slack" required>';
            echo '<small>A friendly name to identify this hook</small>';
            echo '</div>';

            // Webhook URL
            echo '<div class="wvc-hook-field">';
            echo '<label>Webhook URL <span style="color: red;">*</span></label>';
            echo '<input type="url" name="hooks[' . $hook_id . '][webhook_url]" value="' . $webhook_url . '" placeholder="https://hooks.slack.com/services/..." required>';
            echo '<small>Must be an HTTPS URL. Get this from your Slack workspace settings.</small>';
            echo '</div>';

            // Channel override
            echo '<div class="wvc-hook-field">';
            echo '<label>Channel Override (Optional)</label>';
            echo '<input type="text" name="hooks[' . $hook_id . '][channel]" value="' . $channel . '" placeholder="#sales">';
            echo '<small>Leave empty to use the webhook\'s default channel. Use # for channels or @ for DMs.</small>';
            echo '</div>';

            echo '<div class="wvc-hook-status-message" style="display:none;"></div>';
            echo '</div>'; // .wvc-hook-item
        }
    }

    echo '</div>'; // #wvc-hooks-container

    // Add hook button
    echo '<button type="button" class="wvc-add-hook-btn">Add Slack Hook</button>';

    // Submit button
    echo '<div class="submit">';
    echo '<button type="submit" class="button button-primary">💾 Save Hook Settings</button>';
    echo '</div>';

    echo '</form>';
    echo '</div>'; // .wvc-settings-body
    echo '</div>'; // .wvc-form-settings

    // Display the dynamic table
    echo '<div class="wvc-dynamic-table">';
    echo '<table class="wp-list-table widefat fixed striped wvc-submissions-table">';
    echo '<thead>';
    echo '<tr>';
    echo '<th scope="col" style="width: 60px;">#</th>';
    echo '<th scope="col" style="width: 120px;">Submission Date</th>';

    // Add dynamic columns based on form fields
    foreach ($form_fields as $field) {
        // Use proper label from schema, or convert field name as fallback
        $field_label = isset($field_labels[$field]) ? $field_labels[$field] : ucwords(str_replace('_', ' ', $field));
        echo '<th scope="col">' . esc_html($field_label) . '</th>';
    }

    echo '<th scope="col" style="width: 80px;">Actions</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';

    if (empty($submissions)) {
        // Display empty state when no submissions
        $total_columns = 3 + count($form_fields); // # + Date + Fields + Actions
        echo '<tr>';
        echo '<td colspan="' . $total_columns . '" style="text-align: center; padding: 40px 20px; color: #666;">';
        echo '<div style="font-size: 16px; margin-bottom: 10px;">📋</div>';
        echo '<strong>No submissions yet for this form</strong>';
        echo '<p style="margin: 10px 0 0 0; font-size: 14px;">Form fields are ready. Submissions will appear here once the form is submitted.</p>';
        echo '</td>';
        echo '</tr>';
    } else {
        // Display submissions
        $row_number = 1;
        foreach ($submissions as $submission) {
        echo '<tr>';
        echo '<td>' . $row_number . '</td>';
        echo '<td>' . esc_html($submission['submission_date']) . '</td>';

        // Add dynamic field data
        foreach ($form_fields as $field) {
            // Try exact match first, then case-insensitive match
            $field_value = '';
            if (isset($submission['individual_fields'][$field])) {
                $field_value = $submission['individual_fields'][$field];
            } elseif (isset($submission['individual_fields_lowercase_map'][strtolower($field)])) {
                // Fallback to case-insensitive lookup
                $field_value = $submission['individual_fields_lowercase_map'][strtolower($field)];
            }

            // Truncate long values for table display
           /* if (strlen($field_value) > 50) {
                $display_value = substr($field_value, 0, 47) . '...';
            } else {
                $display_value = $field_value;
            }*/

            echo '<td title="' . esc_attr($field_value) . '">' . esc_html($field_value) . '</td>';
        }

        // Add actions column
        // Get post ID directly from submission data
        $post_id = $submission['post_id'] ?? 0;

        echo '<td>';

        if ($post_id > 0) {
            $edit_url = admin_url('admin.php?page=wvc_edit_form_submission&submission_id=' . $post_id . '&form_id=' . urlencode($form_id));
            echo '<a href="' . esc_url($edit_url) . '" class="button button-small button-primary">Edit</a>';
        } else {
            echo '<button class="button button-small button-primary" disabled style="opacity: 0.5; cursor: not-allowed;">Edit</button>';
        }

        echo '</td>';
        echo '</tr>';

        $row_number++;
        }
    }

    echo '</tbody>';
    echo '</table>';
    echo '</div>';

    // Add export functionality
    echo '<div class="wvc-table-actions" style="margin-top: 20px; text-align: right;">';
    echo '<p><strong>Total Submissions:</strong> ' . count($submissions) . '</p>';
    echo '</div>';

    // Display form information at the end
    echo '<div class="wvc-form-info" style="margin: 20px 0; padding: 15px; background: #f8f9fa; border-radius: 5px; border-left: 4px solid #0073aa;">';
    echo '<h3>Form Information</h3>';
    echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">';
    echo '<div><strong>Form ID:</strong> ' . esc_html($form_id) . '</div>';
    echo '<div><strong>Total Submissions:</strong> ' . $submission_count . '</div>';
    echo '<div><strong>Form Fields:</strong> ' . count($form_fields) . '</div>';
    echo '</div>';
    echo '</div>';

    echo '</div>';
}

/**
 * Display edit form submission page
 */
function wvc_edit_form_submission_page() {
    // Get submission ID from URL
    $submission_id = isset($_GET['submission_id']) ? intval($_GET['submission_id']) : 0;
    $form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';

    if (!$submission_id || !$form_id) {
        echo '<div class="wrap">';
        echo '<h1>Edit Submission</h1>';
        echo '<div class="notice notice-error"><p>Invalid submission ID.</p></div>';
        echo '</div>';
        return;
    }

    // Get submission data
    $submission = get_post($submission_id);

    if (!$submission || $submission->post_type !== 'wvc_form_data') {
        echo '<div class="wrap">';
        echo '<h1>Edit Submission</h1>';
        echo '<div class="notice notice-error"><p>Submission not found.</p></div>';
        echo '</div>';
        return;
    }

    // Get form fields from schema
    $form_fields_specs = array();

    // FIRST: Try to get the schema snapshot from the submission itself
    // This ensures we display fields as they existed when the submission was created
    $form_fields_specs = get_post_meta($submission_id, '_wvc_form_fields', true) ?: array();

    // FALLBACK: If no snapshot exists, get from current form (backward compatibility)
    if (empty($form_fields_specs)) {
        $forms = get_posts(array(
            'post_type'   => 'wvc_form',
            'meta_key'    => 'wvc_form_key',
            'meta_value'  => $form_id,
            'post_status' => 'publish',
            'numberposts' => 1,
        ));

        if (!empty($forms)) {
            $form = $forms[0];
            $form_fields_specs = get_post_meta($form->ID, 'wvc_form_fields', true) ?: array();
        }
    }

    // Get current submission data
    $submission_data = wvc_get_form_submission_data($submission_id);
    $current_values = $submission_data['individual_fields'];

    echo '<div class="wrap">';
    echo '<h1>Edit Form Submission</h1>';

    // Breadcrumb navigation
    echo '<p class="wvc-breadcrumb">';
    echo '<a href="' . admin_url('admin.php?page=wvc_form_overview') . '">← Form Overview</a>';
    echo ' | <a href="' . admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id)) . '">Back to ' . esc_html($form_id) . ' Submissions</a>';
    echo '</p>';

    // Show submission info
    echo '<div class="wvc-form-info" style="margin: 20px 0; padding: 15px; background: #f8f9fa; border-radius: 5px; border-left: 4px solid #0073aa;">';
    echo '<h3>Submission Information</h3>';
    echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">';
    echo '<div><strong>Form ID:</strong> ' . esc_html($form_id) . '</div>';
    echo '<div><strong>Submitted:</strong> ' . esc_html($submission_data['submission_date']) . '</div>';
    echo '<div><strong>Submission ID:</strong> #' . $submission_id . '</div>';
    echo '</div>';
    echo '</div>';

    // Edit form
    echo '<form method="post" action="' . admin_url('admin-post.php') . '" class="wvc-edit-submission-form">';
    echo '<input type="hidden" name="action" value="wvc_update_form_submission" />';
    echo '<input type="hidden" name="submission_id" value="' . $submission_id . '" />';
    echo '<input type="hidden" name="form_id" value="' . esc_attr($form_id) . '" />';
    echo wp_nonce_field('wvc_edit_submission_' . $submission_id, 'wvc_edit_nonce', true, false);

    echo '<table class="form-table" role="presentation">';

    // Render fields based on schema if available
    if (!empty($form_fields_specs)) {
        foreach ($form_fields_specs as $field_spec) {
            $field_name = $field_spec['key'] ?? $field_spec['name'] ?? '';
            $field_label = $field_spec['label'] ?? ucwords(str_replace('_', ' ', $field_name));

            // Try to get current value from multiple sources with case-insensitive fallback
            $current_value = '';
            if (isset($current_values[$field_name])) {
                $current_value = $current_values[$field_name];
            } elseif (isset($submission_data['individual_fields_lowercase_map'][strtolower($field_name)])) {
                // Case-insensitive lookup (handles preferLocation -> preferlocation)
                $current_value = $submission_data['individual_fields_lowercase_map'][strtolower($field_name)];
            } elseif (!empty($submission_data['form_data']) && isset($submission_data['form_data'][$field_name])) {
                // Fallback to form_data if individual_fields doesn't have it
                $current_value = $submission_data['form_data'][$field_name];
            } elseif (!empty($submission_data['form_data_complete']) && isset($submission_data['form_data_complete'][$field_name])) {
                // Fallback to form_data_complete
                $current_value = $submission_data['form_data_complete'][$field_name];
            }

            // Add required mark
            $required_mark = !empty($field_spec['required']) ? ' <span class="required" style="color: red;">*</span>' : '';

            echo '<tr>';
            echo '<th scope="row"><label for="field_' . esc_attr($field_name) . '">' . esc_html($field_label) . $required_mark . '</label></th>';
            echo '<td>';

            // Render field using helper function
            wvc_render_submission_field($field_spec, $current_value, $submission_id);

            // Show help text if available
            if (!empty($field_spec['help_text'])) {
                echo '<p class="description" style="margin-top: 3px;">' . esc_html($field_spec['help_text']) . '</p>';
            }

            echo '</td>';
            echo '</tr>';
        }
    } else {
        // Fallback: render all fields from current values
        foreach ($current_values as $field_name => $field_value) {
            $field_label = ucwords(str_replace('_', ' ', $field_name));

            echo '<tr>';
            echo '<th scope="row"><label for="field_' . esc_attr($field_name) . '">' . esc_html($field_label) . '</label></th>';
            echo '<td>';

            // Use textarea for long values
            if (strlen($field_value) > 100) {
                echo '<textarea id="field_' . esc_attr($field_name) . '" name="fields[' . esc_attr($field_name) . ']" class="large-text" rows="5">' . esc_textarea($field_value) . '</textarea>';
            } else {
                echo '<input type="text" id="field_' . esc_attr($field_name) . '" name="fields[' . esc_attr($field_name) . ']" value="' . esc_attr($field_value) . '" class="regular-text" />';
            }

            echo '</td>';
            echo '</tr>';
        }
    }

    echo '</table>';

    // Submit buttons
    echo '<p class="submit">';
    echo '<input type="submit" name="submit" id="submit" class="button button-primary" value="Update Submission" />';
    echo ' <a href="' . admin_url('admin.php?page=wvc_form_submissions&form_id=' . urlencode($form_id)) . '" class="button">Cancel</a>';
    echo ' <button type="button" class="button button-link-delete" style="color: #b32d2e; margin-left: 20px;" onclick="return confirm(\'Are you sure you want to delete this submission?\') && (window.location.href=\'' . wp_nonce_url(admin_url('admin-post.php?action=wvc_delete_form_submission&submission_id=' . $submission_id . '&form_id=' . urlencode($form_id)), 'wvc_delete_submission_' . $submission_id) . '\')">Delete Submission</button>';
    echo '</p>';

    echo '</form>';

    echo '</div>';
}

/**
 * Build HTML5 validation attributes for form submission fields
 *
 * @param string $field_type Field type
 * @param array  $validation Validation rules
 * @param bool   $required   Whether field is required
 * @return string HTML attributes string
 */
function wvc_build_field_validation_attributes($field_type, $validation, $required) {
    $attrs = array();

    if ($required) {
        $attrs[] = 'required';
    }

    if (empty($validation) || !is_array($validation)) {
        return implode(' ', $attrs);
    }

    // Length validation
    if (in_array($field_type, array('text', 'textarea', 'email', 'url', 'tel'))) {
        if (isset($validation['min_length']) && is_numeric($validation['min_length'])) {
            $attrs[] = sprintf('minlength="%d"', intval($validation['min_length']));
        }
        if (isset($validation['max_length']) && is_numeric($validation['max_length'])) {
            $attrs[] = sprintf('maxlength="%d"', intval($validation['max_length']));
        }
    }

    // Numeric validation
    if ($field_type === 'number') {
        if (isset($validation['min']) && is_numeric($validation['min'])) {
            $attrs[] = sprintf('min="%s"', esc_attr($validation['min']));
        }
        if (isset($validation['max']) && is_numeric($validation['max'])) {
            $attrs[] = sprintf('max="%s"', esc_attr($validation['max']));
        }
    }

    // Pattern validation
    if (isset($validation['pattern']) && !empty($validation['pattern'])) {
        $pattern = $validation['pattern'];
        if ($pattern === 'phone') {
            $attrs[] = 'pattern="[0-9\s\-\+\(\)]+"';
            $attrs[] = 'title="Please enter a valid phone number"';
        } elseif (!in_array($pattern, array('email', 'url'))) {
            $attrs[] = sprintf('pattern="%s"', esc_attr($pattern));
        }
    }

    return implode(' ', $attrs);
}

/**
 * Render individual field for form submission editing
 *
 * @param array  $field       The field definition
 * @param mixed  $value       The current field value
 * @param int    $submission_id The submission ID
 */
function wvc_render_submission_field($field, $value, $submission_id) {
    $field_name = $field['key'] ?? $field['name'] ?? '';
    $field_type = $field['type'] ?? 'text';
    $field_id = 'field_' . $field_name;
    $input_name = 'fields[' . $field_name . ']';
    $placeholder = !empty($field['placeholder']) ? 'placeholder="' . esc_attr($field['placeholder']) . '"' : '';
    $validation = isset($field['validation']) ? $field['validation'] : array();

    // Build validation attributes
    $validation_attrs = wvc_build_field_validation_attributes($field_type, $validation, !empty($field['required']));

    switch ($field_type) {
        case 'text':
            echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'textarea':
            echo '<textarea id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" rows="5" ' . $placeholder . ' ' . $validation_attrs . ' class="large-text">' . esc_textarea($value) . '</textarea>';
            break;

        case 'url':
            echo '<input type="url" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_url($value) . '" ' . $placeholder . ' ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'email':
            echo '<input type="email" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'date':
            echo '<input type="date" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'number':
            echo '<input type="number" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'tel':
        case 'phone':
            echo '<input type="tel" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" ' . $placeholder . ' ' . $validation_attrs . ' class="regular-text" />';
            break;

        case 'select':
            echo '<select id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" ' . $validation_attrs . ' class="regular-text">';
            echo '<option value="">Select an option...</option>';
            if (!empty($field['options']) && is_array($field['options'])) {
                foreach ($field['options'] as $option) {
                    $selected = ($value === $option) ? 'selected' : '';
                    echo '<option value="' . esc_attr($option) . '" ' . $selected . '>' . esc_html($option) . '</option>';
                }
            }
            echo '</select>';
            break;

        case 'checkbox':
            $checked = !empty($value) ? 'checked' : '';
            echo '<label for="' . esc_attr($field_id) . '">';
            echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="1" ' . $checked . ' ' . $validation_attrs . ' />';
            echo ' Yes';
            echo '</label>';
            break;

        default:
            echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($input_name) . '" value="' . esc_attr($value) . '" class="regular-text" />';
    }
}

/**
 * Sanitize form submission field value based on field type
 *
 * @param mixed  $value      The field value
 * @param string $field_type The field type
 * @return mixed Sanitized value
 */
function wvc_sanitize_submission_field_value($value, $field_type) {
    switch ($field_type) {
        case 'text':
        case 'select':
            return sanitize_text_field($value);

        case 'textarea':
            return sanitize_textarea_field($value);

        case 'url':
            return esc_url_raw($value);

        case 'email':
            return sanitize_email($value);

        case 'number':
            return intval($value);

        case 'checkbox':
            return !empty($value) ? 1 : 0;

        case 'date':
        case 'tel':
        case 'phone':
            return sanitize_text_field($value);

        default:
            return sanitize_text_field($value);
    }
}
