<?php

namespace WVC\FormSubmissions;

/**
 * WVC Form Submission Email Templates
 *
 * Email template functions for form submission notifications
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate HTML email template for form submission notification
 *
 * @param array $submission_data Complete submission data from wvc_get_form_submission_data()
 * @param string $form_label Form name/label
 * @return string HTML email content
 */
function wvc_generate_submission_email_html($submission_data, $form_label) {
    $form_id = $submission_data['form_id'];
    $submission_id = $submission_data['post_id'];
    $submission_date = $submission_data['submission_date'];
    $fields = $submission_data['individual_fields'];
    $metadata = $submission_data['submission_metadata'] ?? array();

    // Build admin edit URL
    $edit_url = admin_url('admin.php?page=wvc_edit_form_submission&submission_id=' . $submission_id . '&form_id=' . urlencode($form_id));

    // Start HTML template
    ob_start();
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <style>
            body {
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
                line-height: 1.6;
                color: #333;
                background-color: #f4f4f4;
                margin: 0;
                padding: 0;
            }
            .email-container {
                max-width: 600px;
                margin: 20px auto;
                background: #ffffff;
                border-radius: 8px;
                overflow: hidden;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            .email-header {
                background: linear-gradient(135deg, #0073aa 0%, #005177 100%);
                color: #ffffff;
                padding: 30px 20px;
                text-align: center;
            }
            .email-header h1 {
                margin: 0;
                font-size: 24px;
                font-weight: 600;
            }
            .email-body {
                padding: 30px 20px;
            }
            .info-box {
                background: #f8f9fa;
                border-left: 4px solid #0073aa;
                padding: 15px;
                margin-bottom: 25px;
                border-radius: 4px;
            }
            .info-box h2 {
                margin: 0 0 10px 0;
                font-size: 18px;
                color: #0073aa;
            }
            .info-row {
                display: flex;
                padding: 8px 0;
                border-bottom: 1px solid #e1e1e1;
            }
            .info-row:last-child {
                border-bottom: none;
            }
            .info-label {
                font-weight: 600;
                min-width: 150px;
                color: #555;
            }
            .info-value {
                color: #333;
                word-break: break-word;
            }
            .form-fields {
                margin-top: 25px;
            }
            .form-fields h2 {
                font-size: 18px;
                color: #333;
                margin-bottom: 15px;
                padding-bottom: 10px;
                border-bottom: 2px solid #0073aa;
            }
            .field-row {
                padding: 12px;
                margin-bottom: 10px;
                background: #f8f9fa;
                border-radius: 4px;
            }
            .field-label {
                font-weight: 600;
                color: #555;
                font-size: 14px;
                margin-bottom: 5px;
            }
            .field-value {
                color: #333;
                font-size: 15px;
                word-break: break-word;
            }
            .metadata-section {
                margin-top: 25px;
                padding-top: 20px;
                border-top: 2px solid #e1e1e1;
            }
            .metadata-section h3 {
                font-size: 16px;
                color: #555;
                margin-bottom: 10px;
            }
            .button {
                display: inline-block;
                padding: 12px 24px;
                background: #0073aa;
                color: #ffffff !important;
                text-decoration: none;
                border-radius: 4px;
                font-weight: 600;
                margin-top: 20px;
            }
            .email-footer {
                background: #f4f4f4;
                padding: 20px;
                text-align: center;
                font-size: 12px;
                color: #666;
            }
        </style>
    </head>
    <body>
        <div class="email-container">
            <div class="email-header">
                <h1>📋 New Form Submission</h1>
            </div>

            <div class="email-body">
                <div class="info-box">
                    <h2>Submission Details</h2>
                    <div class="info-row">
                        <div class="info-label">Form:</div>
                        <div class="info-value"><?php echo esc_html($form_label); ?></div>
                    </div>
                    <div class="info-row">
                        <div class="info-label">Form ID:</div>
                        <div class="info-value"><?php echo esc_html($form_id); ?></div>
                    </div>
                    <div class="info-row">
                        <div class="info-label">Submission ID:</div>
                        <div class="info-value">#<?php echo esc_html($submission_id); ?></div>
                    </div>
                    <div class="info-row">
                        <div class="info-label">Submitted:</div>
                        <div class="info-value"><?php echo esc_html($submission_date); ?></div>
                    </div>
                </div>

                <div class="form-fields">
                    <h2>Form Data</h2>
                    <?php if (!empty($fields)): ?>
                        <?php foreach ($fields as $field_name => $field_value): ?>
                            <div class="field-row">
                                <div class="field-label">
                                    <?php echo esc_html(ucwords(str_replace('_', ' ', $field_name))); ?>
                                </div>
                                <div class="field-value">
                                    <?php echo nl2br(esc_html($field_value)); ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p>No form data available.</p>
                    <?php endif; ?>
                </div>

                <div class="metadata-section">
                    <h3>Submission Metadata</h3>
                    <div class="info-row">
                        <div class="info-label">Page URL:</div>
                        <div class="info-value">
                            <a href="<?php echo esc_url($submission_data['page_url']); ?>">
                                <?php echo esc_html($submission_data['page_url']); ?>
                            </a>
                        </div>
                    </div>
                    <?php if (!empty($metadata['user_ip'])): ?>
                    <div class="info-row">
                        <div class="info-label">IP Address:</div>
                        <div class="info-value"><?php echo esc_html($metadata['user_ip']); ?></div>
                    </div>
                    <?php endif; ?>
                    <div class="info-row">
                        <div class="info-label">User Agent:</div>
                        <div class="info-value"><?php echo esc_html($submission_data['user_agent']); ?></div>
                    </div>
                </div>

                <div style="text-align: center;">
                    <a href="<?php echo esc_url($edit_url); ?>" class="button">
                        View in WordPress Admin
                    </a>
                </div>
            </div>

            <div class="email-footer">
                <p>This is an automated notification from your WordPress website.</p>
                <p>To manage email notifications, visit your form settings in the admin panel.</p>
            </div>
        </div>
    </body>
    </html>
    <?php
    return ob_get_clean();
}

/**
 * Generate plain text email version (fallback)
 *
 * @param array $submission_data Complete submission data
 * @param string $form_label Form name/label
 * @return string Plain text email content
 */
function wvc_generate_submission_email_text($submission_data, $form_label) {
    $form_id = $submission_data['form_id'];
    $submission_id = $submission_data['post_id'];
    $submission_date = $submission_data['submission_date'];
    $fields = $submission_data['individual_fields'];
    $metadata = $submission_data['submission_metadata'] ?? array();

    $edit_url = admin_url('admin.php?page=wvc_edit_form_submission&submission_id=' . $submission_id . '&form_id=' . urlencode($form_id));

    $text = "NEW FORM SUBMISSION\n";
    $text .= str_repeat('=', 50) . "\n\n";

    $text .= "SUBMISSION DETAILS\n";
    $text .= "Form: {$form_label}\n";
    $text .= "Form ID: {$form_id}\n";
    $text .= "Submission ID: #{$submission_id}\n";
    $text .= "Submitted: {$submission_date}\n\n";

    $text .= str_repeat('-', 50) . "\n";
    $text .= "FORM DATA\n";
    $text .= str_repeat('-', 50) . "\n\n";

    if (!empty($fields)) {
        foreach ($fields as $field_name => $field_value) {
            $label = ucwords(str_replace('_', ' ', $field_name));
            $text .= "{$label}:\n{$field_value}\n\n";
        }
    } else {
        $text .= "No form data available.\n\n";
    }

    $text .= str_repeat('-', 50) . "\n";
    $text .= "METADATA\n";
    $text .= str_repeat('-', 50) . "\n\n";
    $text .= "Page URL: {$submission_data['page_url']}\n";
    if (!empty($metadata['user_ip'])) {
        $text .= "IP Address: {$metadata['user_ip']}\n";
    }
    $text .= "User Agent: {$submission_data['user_agent']}\n\n";

    $text .= str_repeat('=', 50) . "\n";
    $text .= "View in WordPress Admin:\n{$edit_url}\n";

    return $text;
}
