<?php
if ( ! defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * WVC Images Upload
 *
 * Basic class for uploading images
 *
 * @package WVC_Theme
 * @subpackage Import
 * @since 1.0.0
 */
class WVC_Images_Upload
{
    private $images_data;
    /**
     * @param $images_data
     */
    public function __construct($images_data)
    {
        $this->images_data = $images_data;
        // Include SVG upload handler
        $this->includeSvgUploadHandler();

        // Initialize SVG support for the import process
        $this->enableSvgSupport();
    }

    public function getImagesData()
    {
        return $this->images_data;
    }

    public function createImages()
    {
        $result = [
            'created' => 0,
            'updated' => 0,
            'failed'  => 0,
            'images'  => []
        ];

        if ( ! isset($this->images_data) || empty($this->images_data)) {
            return $result;
        }

        $images_data = $this->images_data;

        foreach ($images_data as $image_key => $image_data) {
            try {
                // Skip if already has wp_src (already imported)
                if ( ! empty($image_data['wp_src'])) {
                    $result['updated']++;
                    continue;
                }

                // Check if cloud_src exists
                if (empty($image_data['cloud_src'])) {
                    $result['failed']++;
                    $result['images'][$image_key] = [
                        'status'         => 'failed',
                        'error'          => 'No cloud_src URL provided',
                        'attachment_id'  => "",
                        'attachment_url' => ""
                    ];
                    continue;
                }

                // check if there is an image in the wp media library with the same cloud_src
                $attachment_id = $this->getImageIdByCloudSrc($image_data['cloud_src']);

                if ($attachment_id) {
                    $result['updated']++;
                    $attachment_url               = wp_get_attachment_url($attachment_id);
                    $result['images'][$image_key] = [
                        'status'         => 'updated',
                        'attachment_id'  => $attachment_id,
                        'attachment_url' => $attachment_url
                    ];

                    // replace the image url in the content with the wp_src
                    $this->images_data[$image_key]['wp_src'] = $attachment_url;
                    $this->images_data[$image_key]['wp_id']  = $attachment_id;

                    continue;
                }

                // Download image from S3
                $downloaded_file = $this->downloadImageFromS3($image_data['cloud_src']);

                if (is_wp_error($downloaded_file)) {
                    $result['failed']++;
                    $result['images'][$image_key] = [
                        'status'         => 'failed',
                        'error'          => $downloaded_file->get_error_message(),
                        'attachment_id'  => "",
                        'attachment_url' => ""
                    ];
                    continue;
                }

                // Import to WordPress media library
                $attachment_id = $this->importImageToMediaLibrary($downloaded_file, $image_data);

                if (is_wp_error($attachment_id)) {
                    $result['failed']++;
                    $result['images'][$image_key] = [
                        'status'         => 'failed',
                        'attachment_id'  => "",
                        'attachment_url' => "",
                        'error'          => $attachment_id->get_error_message()
                    ];
                    // Clean up downloaded file
                    @unlink($downloaded_file);
                    continue;
                }

                // Get the attachment URL
                $attachment_url = wp_get_attachment_url($attachment_id);

                $this->assignImageToPostTerm($attachment_id, $image_data);

                // Update the images_data with WordPress post ID and URL
                $this->images_data[$image_key]['wp_src'] = $attachment_url;
                $this->images_data[$image_key]['wp_id']  = $attachment_id;

                $result['created']++;
                $result['images'][$image_key] = [
                    'status'         => 'success',
                    'attachment_id'  => $attachment_id,
                    'attachment_url' => $attachment_url
                ];
            } catch (Exception $e) {
                $result['failed']++;
                $result['images'][$image_key] = [
                    'status' => 'failed',
                    'error'  => $e->getMessage()
                ];
            }
        }

        return $result;
    }


    /**
     * Enable SVG support for the current import
     */
    public function enableSvgSupport()
    {
        // Initialize SVG upload handler if it exists
        if (class_exists('WVC_SVG_Upload_Handler')) {
            WVC_SVG_Upload_Handler::getInstance();
        }

        // Add SVG mime type to allowed uploads
        add_filter('upload_mimes', function ($mimes) {
            $mimes['svg'] = 'image/svg+xml';

            return $mimes;
        });

        // Handle SVG file type checking
        add_filter('wp_check_filetype_and_ext', function ($data, $file, $filename, $mimes) {
            $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            if ($extension === 'svg') {
                $data['type'] = 'image/svg+xml';
                $data['ext']  = 'svg';
            }

            return $data;
        }, 10, 4);

        // Ensure SVG uploads are allowed for this request
        add_filter('wp_handle_upload_prefilter', function ($file) {
            if ($file['type'] === 'image/svg+xml') {
                // Allow SVG uploads by ensuring no error is set
                if (isset($file['error']) && $file['error'] !== 0) {
                    unset($file['error']);
                }
            }

            return $file;
        });
    }

    /**
     * Include SVG upload handler
     */
    private function includeSvgUploadHandler()
    {
        $svg_handler_path = __DIR__ . '/svg-upload-handler.php';
        if (file_exists($svg_handler_path)) {
            require_once $svg_handler_path;
        }

        // Include SVG sanitizer if it exists
        $svg_sanitizer_path = __DIR__ . '/svg-sanitizer.php';
        if (file_exists($svg_sanitizer_path)) {
            require_once $svg_sanitizer_path;
        }
    }

    /**
     * @param $cloud_src
     *
     * @return mixed|null
     */
    private function getImageIdByCloudSrc($cloud_src)
    {
        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'meta_query'     => [
                [
                    'key'     => '_wvc_cloud_src',
                    'value'   => $cloud_src,
                    'compare' => '='
                ]
            ],
            'posts_per_page' => 1,
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC'
        ];

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            return $query->posts[0];
        }

        return null;
    }

    /**
     * Import image to WordPress media library
     *
     * @param string $file_path The path to the image file
     * @param array $image_data The image data from the JSON
     *
     * @return int|WP_Error Attachment ID or WP_Error on failure
     */
    private function importImageToMediaLibrary($file_path, $image_data)
    {
        // Check if file exists and is readable
        if ( ! file_exists($file_path) || ! is_readable($file_path)) {
            return new WP_Error('file_not_found', 'Image file not found or not readable');
        }

        // Enable SVG support for this import
        $this->enableSvgSupport();

        // Get file info using our enhanced method
        $file_info = $this->getFileInfo($file_path);

        // Handle SVG files specifically
        if ($file_info['type'] === 'image/svg+xml') {
            // Sanitize SVG file before upload
            $svg_sanitized = $this->sanitizeSvgFile($file_path);
            if (is_wp_error($svg_sanitized)) {
                return $svg_sanitized;
            }
        } elseif (empty($file_info['type'])) {
            return new WP_Error('invalid_file_type', 'Invalid file type');
        }

        // For SVG files, we need to ensure proper handling through WordPress upload system
        if ($file_info['type'] === 'image/svg+xml') {
            // Create a temporary file array that WordPress can process
            $file_array = [
                'name'     => basename($image_data['src']),
                'tmp_name' => $file_path,
                'type'     => 'image/svg+xml',
                'error'    => 0,
                'size'     => filesize($file_path)
            ];

            // Use wp_handle_sideload with SVG-specific handling
            $upload = wp_handle_sideload($file_array, ['test_form' => false]);

            if (isset($upload['error'])) {
                return new WP_Error('upload_failed', $upload['error']);
            }

            // Prepare attachment post data
            $attachment = [
                'post_title'     => ! empty($image_data['description']) ? $image_data['description'] : basename($image_data['src']),
                'post_content'   => ! empty($image_data['prompt']) ? $image_data['prompt'] : '',
                'post_excerpt'   => ! empty($image_data['description']) ? $image_data['description'] : '',
                'post_status'    => 'inherit',
                'post_mime_type' => 'image/svg+xml'
            ];

            // Insert the attachment
            $attachment_id = wp_insert_attachment($attachment, $upload['file']);

            if (is_wp_error($attachment_id)) {
                return $attachment_id;
            }

            // For SVG files, we don't generate metadata as WordPress doesn't support it
            // But we can add custom metadata if needed

        } else {
            // Handle regular image files as before
            $file_array = [
                'name'     => basename($image_data['src']),
                'tmp_name' => $file_path,
                'type'     => $file_info['type'],
                'error'    => 0,
                'size'     => filesize($file_path)
            ];

            // Use wp_handle_sideload to move the file to the uploads directory
            $upload = wp_handle_sideload($file_array, ['test_form' => false]);

            if (isset($upload['error'])) {
                return new WP_Error('upload_failed', $upload['error']);
            }

            // Prepare attachment post data
            $attachment = [
                'post_title'     => ! empty($image_data['description']) ? $image_data['description'] : basename($image_data['src']),
                'post_content'   => ! empty($image_data['prompt']) ? $image_data['prompt'] : '',
                'post_excerpt'   => ! empty($image_data['description']) ? $image_data['description'] : '',
                'post_status'    => 'inherit',
                'post_mime_type' => $file_info['type']
            ];

            // Insert the attachment
            $attachment_id = wp_insert_attachment($attachment, $upload['file']);

            if (is_wp_error($attachment_id)) {
                return $attachment_id;
            }

            // Generate attachment metadata for non-SVG files
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            $attachment_data = wp_generate_attachment_metadata($attachment_id, $upload['file']);
            wp_update_attachment_metadata($attachment_id, $attachment_data);
        }

        // Store cloud_src as metadata if it exists
        if ( ! empty($image_data['cloud_src'])) {
            update_post_meta($attachment_id, '_wvc_cloud_src', $image_data['cloud_src']);
        }

        return $attachment_id;
    }

    /**
     * Get proper file info for SVG files
     *
     * @param string $file_path Path to the file
     *
     * @return array File info array
     */
    private function getFileInfo($file_path)
    {
        $file_info = wp_check_filetype(basename($file_path), null);

        // If WordPress couldn't determine the type, check manually
        if (empty($file_info['type'])) {
            $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));

            if ($extension === 'svg') {
                $file_info = [
                    'type' => 'image/svg+xml',
                    'ext'  => 'svg'
                ];
            }
        }

        return $file_info;
    }

    /**
     * Sanitize SVG file before upload
     *
     * @param string $file_path Path to the SVG file
     *
     * @return true|WP_Error True on success, WP_Error on failure
     */
    private function sanitizeSvgFile($file_path)
    {
        // Check if SVG sanitizer class exists
        if ( ! class_exists('WVC_SVG_Sanitizer')) {
            // If no sanitizer is available, we'll proceed without sanitization
            // but log a warning
            error_log('WVC Import: SVG sanitizer not found, proceeding without sanitization');

            return true;
        }

        // Read the file content
        $file_content = file_get_contents($file_path);
        if (false === $file_content) {
            return new WP_Error('svg_read_failed', 'Could not read SVG file');
        }

        // Sanitize the SVG content
        $sanitizer         = new WVC_SVG_Sanitizer();
        $sanitized_content = $sanitizer->sanitize_file($file_content);

        if (null === $sanitized_content) {
            return new WP_Error('svg_sanitize_failed', 'SVG file could not be sanitized');
        }

        // Write the sanitized content back to the file
        $result = file_put_contents($file_path, $sanitized_content);
        if (false === $result) {
            return new WP_Error('svg_write_failed', 'Could not write sanitized SVG content');
        }

        return true;
    }

    private function assignImageToPostTerm($attachment_id, $image_data)
    {
        if (empty($attachment_id) || empty($image_data) || empty($image_data['used_in']) || ! is_array($image_data['used_in'])) {
            return;
        }

        $used_in = $image_data['used_in'];

        if (isset($used_in['posts_featured']) && ! empty($used_in['posts_featured'])) {
            $post_targets = $this->extractImageAssignmentTargets($used_in['posts_featured']);

            foreach ($post_targets as $target) {
                $post_id = $this->resolvePostIdFromImageTarget($target);

                if ($post_id) {
                    set_post_thumbnail($post_id, $attachment_id);
                }
            }
        }

        if (isset($used_in['terms']) && ! empty($used_in['terms']) && is_array($used_in['terms'])) {
            foreach ($used_in['terms'] as $taxonomy => $term_data) {
                $term_targets = $this->extractImageAssignmentTargets($term_data);

                foreach ($term_targets as $target) {
                    $term_id = $this->resolveTermIdFromImageTarget($target, $taxonomy);

                    if ($term_id) {
                        update_term_meta($term_id, 'thumbnail_id', $attachment_id);
                    }
                }
            }
        }
    }

    private function extractImageAssignmentTargets($data)
    {
        $targets = [];

        if ( ! is_array($data)) {
            return $targets;
        }

        $has_identifiers = (isset($data['wpid']) && $data['wpid']) || (isset($data['uid']) && $data['uid'] !== '');

        if ($has_identifiers) {
            $targets[] = $data;

            return $targets;
        }

        foreach ($data as $value) {
            if (is_array($value)) {
                $targets = array_merge($targets, $this->extractImageAssignmentTargets($value));
            }
        }

        return $targets;
    }

    private function resolvePostIdFromImageTarget($target)
    {
        if (isset($target['wpid']) && $target['wpid']) {
            $post = get_post((int)$target['wpid']);

            if ($post instanceof WP_Post) {
                return (int)$post->ID;
            }
        }

        if (isset($target['uid']) && $target['uid'] !== '') {
            $uid = sanitize_text_field($target['uid']);

            $posts = get_posts([
                'post_type'      => 'any',
                'post_status'    => 'any',
                'posts_per_page' => 1,
                'fields'         => 'ids',
                'meta_query'     => [
                    [
                        'key'     => 'wvc_uid',
                        'value'   => $uid,
                        'compare' => '='
                    ]
                ]
            ]);

            if ( ! empty($posts)) {
                return (int)$posts[0];
            }
        }

        return null;
    }

    private function resolveTermIdFromImageTarget($target, $taxonomy)
    {
        if (isset($target['wpid']) && $target['wpid']) {
            $term = get_term((int)$target['wpid'], $taxonomy);

            if ($term && ! is_wp_error($term)) {
                return (int)$term->term_id;
            }
        }

        if (isset($target['uid']) && $target['uid'] !== '') {
            $uid = sanitize_text_field($target['uid']);

            $terms = get_terms([
                'taxonomy'               => $taxonomy,
                'hide_empty'             => false,
                'number'                 => 1,
                'fields'                 => 'ids',
                'meta_query'             => [
                    [
                        'key'     => 'wvc_uid',
                        'value'   => $uid,
                        'compare' => '='
                    ]
                ],
                'update_term_meta_cache' => false
            ]);

            if ( ! is_wp_error($terms) && ! empty($terms)) {
                return (int)$terms[0];
            }
        }

        return null;
    }


    /**
     * Download image from S3 URL
     *
     * @param string $s3_url The S3 URL to download from
     *
     * @return string|WP_Error Temporary file path or WP_Error on failure
     */
    private function downloadImageFromS3($s3_url)
    {

        // Extract filename and extension from URL
        $url_parts = parse_url($s3_url);
        $path      = isset($url_parts['path']) ? $url_parts['path'] : '';
        $filename  = pathinfo($path, PATHINFO_FILENAME);
        $extension = pathinfo($path, PATHINFO_EXTENSION);

        // If no filename found, use unique ID
        if (empty($filename)) {
            $uid = uniqid();

            $filename = 'wvc_image_' . $uid;
        }

        // If no extension found, default to .png
        if (empty($extension)) {
            $extension = 'png';
        }

        // Create a temporary file with correct filename and extension
        $temp_file = wp_tempnam($filename);

        if ( ! $temp_file) {
            return new WP_Error('temp_file_failed', 'Failed to create temporary file');
        }

        // Rename the temp file to have the correct extension
        $temp_file_with_ext = str_replace('.tmp', '.' . $extension, $temp_file);
        if (rename($temp_file, $temp_file_with_ext)) {
            $temp_file = $temp_file_with_ext;
        }

        // Use WordPress HTTP API to download the file directly to temp file
        $response = wp_remote_get($s3_url, [
            'timeout'  => 60,
            'stream'   => true,
            'filename' => $temp_file
        ]);

        if (is_wp_error($response)) {
            @unlink($temp_file);

            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            @unlink($temp_file);

            return new WP_Error('download_failed', sprintf('Failed to download image. HTTP code: %d', $response_code));
        }

        // Check if the file was actually downloaded and has content
        if ( ! file_exists($temp_file) || filesize($temp_file) === 0) {
            @unlink($temp_file);

            return new WP_Error('empty_response', 'Empty response from S3 or file not downloaded');
        }

        return $temp_file;
    }


}