<?php
/**
 * Media Operations REST API Class
 *
 * Provides REST API endpoints for uploading, retrieving, updating, and deleting media files.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/post-manager.php';
require_once get_template_directory() . '/includes/import/images-upload.php';

/**
 * Class WVC_Media_REST_API
 *
 * Handles media operations via REST API
 */
class WVC_Media_REST_API extends WVC_REST_API
{

    /**
     * Post Manager instance (for media operations)
     *
     * @var WVC_Post_Manager
     */
    private $post_manager;

    /**
     * Required capability for media management
     */
    const UPLOAD_FILES_CAP = 'upload_files';

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->post_manager = WVC_Post_Manager::get_instance();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Register media endpoints
        $this->register_media_routes();
    }

    /**
     * Register media-related routes
     */
    private function register_media_routes()
    {
        // Upload media endpoint
        register_rest_route(
            $this->namespace,
            '/media',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'upload_media'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::UPLOAD_FILES_CAP);
                },
                'args'                => array(
                    'post_id'     => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('The post ID to attach the media to', 'wvc-theme'),
                        'default'     => 0,
                    ),
                    'title'       => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The title of the media', 'wvc-theme'),
                    ),
                    'caption'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The caption of the media', 'wvc-theme'),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The description of the media', 'wvc-theme'),
                    ),
                    'alt_text'    => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The alternative text for the media', 'wvc-theme'),
                    ),
                    "featured_in" => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'posts_ids' => array(
                                'type' => 'integer',
                            ),
                            "terms_ids" => array(
                                'type' => 'integer',
                            ),
                        ),
                        'description' => __('The posts and terms to feature the media in', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Get media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_media'),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the media to retrieve', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Update media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'update_media'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::UPLOAD_FILES_CAP);
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the media to update', 'wvc-theme'),
                    ),
                    'title'       => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The title of the media', 'wvc-theme'),
                    ),
                    'caption'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The caption of the media', 'wvc-theme'),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The description of the media', 'wvc-theme'),
                    ),
                    'alt_text'    => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The alternative text for the media', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Delete media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'delete_media'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::UPLOAD_FILES_CAP);
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the media to delete', 'wvc-theme'),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __('Whether to bypass trash and force deletion', 'wvc-theme'),
                    ),
                ),
            )
        );

        register_rest_route(
            $this->namespace,
            '/media-bulk',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'upload_media_bulk'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::UPLOAD_FILES_CAP);
                },
                'args'                => array(
                    'images_data' => array(
                        'required'    => false,
                        'type'        => 'array',
                        'description' => __('Images data', 'wvc-theme'),
                        'default'     => 0,
                    )
                ),
            )
        );
    }

    /**
     * Upload media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function upload_media($request)
    {
        // Handle binary upload from Python client (raw data)
        $content_type        = $request->get_header('content-type');
        $content_disposition = $request->get_header('content-disposition');

        // Check if we have proper headers for binary upload
        if (empty($content_disposition)) {
            return new WP_Error(
                'missing_headers',
                __('Missing Content-Disposition header', 'wvc-theme'),
                array('status' => 400)
            );
        }

        // Parse filename from Content-Disposition header
        $filename = '';
        if (preg_match('/filename="([^"]+)"/', $content_disposition, $matches)) {
            $filename = $matches[1];
        } else {
            return new WP_Error(
                'invalid_filename',
                __('Invalid or missing filename in Content-Disposition header', 'wvc-theme'),
                array('status' => 400)
            );
        }

        // Get the raw data from the request
        $image_data = $request->get_body();

        if (empty($image_data)) {
            return new WP_Error(
                'empty_data',
                __('No data was provided in the request body', 'wvc-theme'),
                array('status' => 400)
            );
        }

        // Get the post ID if provided
        $post_id = $request->get_param('post_id');

        // Set up arguments for media upload
        $args = array(
            'title'       => $request->get_param('title'),
            'caption'     => $request->get_param('caption'),
            'description' => $request->get_param('description'),
            'alt_text'    => $request->get_param('alt_text'),
        );

        // Use the refactored upload_media method
        $result = $this->post_manager->upload_media(
            $image_data,
            $filename,
            $content_type,
            $post_id,
            $args
        );

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Get media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_media($request)
    {
        $media_id = $request->get_param('id');
        $result   = $this->post_manager->get_media($media_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Update media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_media($request)
    {
        $media_id = $request->get_param('id');
        $args     = array(
            'title'       => $request->get_param('title'),
            'caption'     => $request->get_param('caption'),
            'description' => $request->get_param('description'),
            'alt_text'    => $request->get_param('alt_text'),
        );

        $result = $this->post_manager->update_media($media_id, $args);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Delete media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_media($request)
    {
        $media_id = $request->get_param('id');
        $force    = $request->get_param('force');

        $result = $this->post_manager->delete_media($media_id, $force);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    public function upload_media_bulk($request)
    {
        // Prevent PHP timeout for long-running bulk uploads
        // Set execution time to 4 minutes (240 seconds)
        @set_time_limit(240);
        @ini_set('max_execution_time', 240);
        

        $media_data = $request->get_param('images_data');

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';

        $image_upload = new WVC_Images_Upload($media_data);
        $result       = $image_upload->createImages();

        return rest_ensure_response($result);

    }
}

// Initialize the Media Operations REST API
new WVC_Media_REST_API();
