<?php
/**
 * Page Operations REST API Class
 *
 * Provides REST API endpoints for creating and deleting WordPress pages.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/page-manager.php';

/**
 * Class WVC_Page_REST_API
 *
 * Handles page creation and deletion via REST API
 */
class WVC_Page_REST_API extends WVC_REST_API
{

    /**
     * Page Manager instance
     *
     * @var WVC_Page_Manager
     */
    private $page_manager;

    const EDIT_PAGES_CAP = "edit_pages";

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->page_manager = WVC_Page_Manager::get_instance();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Create page endpoint
        register_rest_route(
            $this->namespace,
            '/pages',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'create_page'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_PAGES_CAP);
                },
                'args'                => array(
                    'title'            => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The title of the page', 'wvc-theme'),
                    ),
                    'content-managers' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The content-managers of the page', 'wvc-theme'),
                    ),
                    'template'         => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The template to use for the page', 'wvc-theme'),
                    ),
                    'status'           => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'default'           => 'publish',
                        'enum'              => array('publish', 'draft', 'pending', 'private'),
                        'description'       => __('The status of the page', 'wvc-theme'),
                    ),
                    'meta'             => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __('Meta data for the page', 'wvc-theme'),
                    ),
                    'featured_image'   => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('Featured image ID for the page', 'wvc-theme'),
                    ),
                    'parent'           => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('Parent page ID', 'wvc-theme'),
                    ),
                    'menu_order'       => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('Order the page should appear in', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Get page endpoint
        register_rest_route(
            $this->namespace,
            '/pages/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_page'),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the page to retrieve', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Edit page endpoint
        register_rest_route(
            $this->namespace,
            '/pages/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'edit_page'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_PAGES_CAP);
                },
                'args'                => array(
                    'id'               => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the page to edit', 'wvc-theme'),
                    ),
                    'title'            => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The new title for the page', 'wvc-theme'),
                    ),
                    'content-managers' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('The new content-managers for the page', 'wvc-theme'),
                    ),
                    'status'           => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'enum'              => array('publish', 'draft', 'pending', 'private'),
                        'description'       => __('The new status for the page', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Delete page endpoint
        register_rest_route(
            $this->namespace,
            '/pages/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'delete_page'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_PAGES_CAP);
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the page to delete', 'wvc-theme'),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __('Whether to bypass trash and force deletion', 'wvc-theme'),
                    ),
                ),
            )
        );
    }

    /**
     * Create a new page
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_page($request)
    {
        $args = array(
            'title'            => $request->get_param('title'),
            'content-managers' => $request->get_param('content-managers'),
            'template'         => $request->get_param('template'),
            'status'           => $request->get_param('status'),
            'meta'             => $request->get_param('meta'),
            'featured_image'   => $request->get_param('featured_image'),
            'parent'           => $request->get_param('parent'),
            'menu_order'       => $request->get_param('menu_order'),
        );

        $result = $this->page_manager->create_page($args);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Get a page
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_page($request)
    {
        $page_id = $request->get_param('id');
        $result  = $this->page_manager->get_page($page_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Edit a page
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function edit_page($request)
    {
        $page_id = $request->get_param('id');
        $args    = array(
            'title'            => $request->get_param('title'),
            'content-managers' => $request->get_param('content-managers'),
            'status'           => $request->get_param('status'),
            'template'         => $request->get_param('template'),
            'meta'             => $request->get_param('meta'),
            'featured_image'   => $request->get_param('featured_image'),
            'parent'           => $request->get_param('parent'),
            'menu_order'       => $request->get_param('menu_order'),
        );

        $result = $this->page_manager->edit_page($page_id, $args);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Delete a page
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_page($request)
    {
        $page_id = $request->get_param('id');
        $force   = $request->get_param('force');

        $result = $this->page_manager->delete_page($page_id, $force);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }
}

// Initialize the Page Operations REST API
new WVC_Page_REST_API();