<?php
/**
 * WVC Plugin REST API
 *
 * @since      1.0.0
 * @package    WVC_Editor
 * @subpackage WVC_Editor/REST
 */

// Exit if accessed directly
if ( ! defined('ABSPATH')) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/plugin-manager.php';

/**
 * WVC Plugin REST API Class
 */
class WVC_Plugin_REST_API extends WVC_REST_API
{
    const INSTALL_PLUGIN_CAP = "install_plugin";
    const UNINSTALL_PLUGIN_CAP = "uninstall_plugin";

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Install plugin endpoint
        register_rest_route('wvc/v1', '/plugin/install', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [$this, 'install_plugin'],
            'permission_callback' =>
                function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::INSTALL_PLUGIN_CAP);
                }
            ,
            'args'                => [
                'plugin_slug' => [
                    'required'          => true,
                    'type'              => 'string',
                    'description'       => 'Plugin slug to install',
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);

        // Uninstall plugin endpoint
        register_rest_route('wvc/v1', '/plugin/uninstall', [
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => [$this, 'uninstall_plugin'],
            'permission_callback' =>
                function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::UNINSTALL_PLUGIN_CAP);
                }
            ,
            'args'                => [
                'plugin_slug' => [
                    'required'          => true,
                    'type'              => 'string',
                    'description'       => 'Plugin slug to uninstall',
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);

        // List plugins endpoint
        register_rest_route('wvc/v1', '/plugin/list', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [$this, 'list_plugins'],
            'permission_callback' => [$this, 'permissions_check']
        ]);
    }

    /**
     * Install plugin
     */
    public function install_plugin($request)
    {
        $plugin_slug    = $request->get_param('plugin_slug');
        $plugin_manager = WVC_Plugin_Manager::get_instance();
        $result         = $plugin_manager->install_plugin($plugin_slug);

        $status_code = $result['success'] ? 200 : 500;

        return new WP_REST_Response($result, $status_code);
    }

    /**
     * Uninstall plugin
     */
    public function uninstall_plugin($request)
    {
        $plugin_slug    = $request->get_param('plugin_slug');
        $plugin_manager = WVC_Plugin_Manager::get_instance();
        $result         = $plugin_manager->uninstall_plugin($plugin_slug);

        $status_code = $result['success'] ? 200 : 404;

        return new WP_REST_Response($result, $status_code);
    }

    /**
     * List installed plugins
     */
    public function list_plugins($request)
    {
        $plugin_manager = WVC_Plugin_Manager::get_instance();
        $result         = $plugin_manager->list_plugins();

        $status_code = $result['success'] ? 200 : 500;

        return new WP_REST_Response($result, $status_code);
    }


}

// Initialize the Plugin REST API
new WVC_Plugin_REST_API();