<?php
/**
 * Term REST API Class
 *
 * Provides REST API endpoints for term management across any taxonomy.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/term-manager.php';

/**
 * Class WVC_Term_REST_API
 *
 * Handles REST API endpoints for term management
 */
class WVC_Term_REST_API extends WVC_REST_API {

    /**
     * Namespace for the REST API
     *
     * @var string
     */
    protected $namespace = "wvc/v1";

    /**
     * Instance of this singleton class
     *
     * @var WVC_Term_REST_API
     */
    private static $instance = null;

    /**
     * Get instance of this class
     *
     * @return WVC_Term_REST_API
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( "rest_api_init", array( $this, "register_routes" ) );
    }

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Create term
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)", array(
            "methods"             => WP_REST_Server::CREATABLE,
            "callback"            => array( $this, "create_term" ),
            "permission_callback" => array( $this, "create_term_permission_check" ),
            "args"                => $this->get_create_term_args(),
        ) );

        // Bulk create/update terms
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)/bulk", array(
            "methods"             => WP_REST_Server::CREATABLE,
            "callback"            => array( $this, "bulk_upsert_terms" ),
            "permission_callback" => array( $this, "bulk_upsert_terms_permission_check" ),
            "args"                => array(
                "taxonomy" => array(
                    "required"          => true,
                    "type"              => "string",
                    "sanitize_callback" => "sanitize_text_field",
                ),
            ),
        ) );

        // Get term
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)/(?P<term_id>\d+)", array(
            "methods"             => WP_REST_Server::READABLE,
            "callback"            => array( $this, "get_term" ),
            "permission_callback" => array( $this, "get_term_permission_check" ),
            "args"                => array(
                "taxonomy" => array(
                    "required"          => true,
                    "type"              => "string",
                    "sanitize_callback" => "sanitize_text_field",
                ),
                "term_id"  => array(
                    "required"          => true,
                    "type"              => "integer",
                    "sanitize_callback" => "absint",
                ),
            ),
        ) );

        // Update term
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)/(?P<term_id>\d+)", array(
            "methods"             => WP_REST_Server::EDITABLE,
            "callback"            => array( $this, "update_term" ),
            "permission_callback" => array( $this, "update_term_permission_check" ),
            "args"                => $this->get_update_term_args(),
        ) );

        // Delete term
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)/(?P<term_id>\d+)", array(
            "methods"             => WP_REST_Server::DELETABLE,
            "callback"            => array( $this, "delete_term" ),
            "permission_callback" => array( $this, "delete_term_permission_check" ),
            "args"                => array(
                "taxonomy" => array(
                    "required"          => true,
                    "type"              => "string",
                    "sanitize_callback" => "sanitize_text_field",
                ),
                "term_id"  => array(
                    "required"          => true,
                    "type"              => "integer",
                    "sanitize_callback" => "absint",
                ),
            ),
        ) );

        // List terms
        register_rest_route( $this->namespace, "/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)", array(
            "methods"             => WP_REST_Server::READABLE,
            "callback"            => array( $this, "list_terms" ),
            "permission_callback" => array( $this, "list_terms_permission_check" ),
            "args"                => $this->get_list_terms_args(),
        ) );
    }

    /**
     * Create a new term
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function create_term( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $args = $request->get_json_params();

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $result = $term_manager->create_term( $taxonomy, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Bulk create or update terms
     *
     * Accepts a JSON object (dictionary) where each key maps to a term data object.
     * Each term data object may include an "id" (or "term_id"). If an id is present,
     * the term is updated; otherwise, a new term is created. Returns a dictionary
     * mapping the same input keys to resulting term IDs.
     *
     * Also accepts an envelope object with a top-level "terms" dictionary for flexibility.
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_upsert_terms( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $payload  = $request->get_json_params();

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        // Normalize input: accept either a root-level dict, or an envelope with "terms" dict
        $items = array();
        if ( is_array( $payload ) ) {
            if ( array_key_exists( "terms", $payload ) && is_array( $payload["terms"] ) ) {
                $items = $payload["terms"];
            } else {
                $items = $payload;
            }
        }

        if ( empty( $items ) || ! is_array( $items ) ) {
            return new WP_Error(
                "invalid_body",
                __( "Request body must be an object mapping keys to term data", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $ids_by_key = array();

        foreach ( $items as $key => $term_data ) {
            if ( ! is_array( $term_data ) ) {
                $ids_by_key[ (string) $key ] = 0;
                continue;
            }

            $id = null;
            if ( isset( $term_data["id"] ) ) {
                $id = absint( $term_data["id"] );
                unset( $term_data["id"] );
            } elseif ( isset( $term_data["term_id"] ) ) {
                $id = absint( $term_data["term_id"] );
                unset( $term_data["term_id"] );
            }

            if ( $id ) {
                // Update path: on error, return the same id
                $result = $term_manager->update_term( $id, $taxonomy, $term_data );
                if ( is_wp_error( $result ) ) {
                    $ids_by_key[ (string) $key ] = (int) $id;
                } else {
                    $ids_by_key[ (string) $key ] = isset( $result["id"] ) ? (int) $result["id"] : (int) $id;
                }
            } else {
                // Create path: on error, return 0
                $result = $term_manager->create_term( $taxonomy, $term_data );
                if ( is_wp_error( $result ) ) {
                    $ids_by_key[ (string) $key ] = 0;
                } else {
                    $ids_by_key[ (string) $key ] = isset( $result["id"] ) ? (int) $result["id"] : 0;
                }
            }
        }

        return rest_ensure_response( $ids_by_key );
    }

    /**
     * Get a term
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function get_term( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $term_id = $request->get_param( "term_id" );

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $result = $term_manager->get_term( $term_id, $taxonomy );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a term
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function update_term( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $term_id = $request->get_param( "term_id" );
        $args = $request->get_json_params();

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $result = $term_manager->update_term( $term_id, $taxonomy, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a term
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function delete_term( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $term_id = $request->get_param( "term_id" );

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $result = $term_manager->delete_term( $term_id, $taxonomy );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * List terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return WP_REST_Response|WP_Error
     */
    public function list_terms( $request ) {
        $taxonomy = $request->get_param( "taxonomy" );
        $per_page = $request->get_param( "per_page" );
        $page = $request->get_param( "page" );
        $search = $request->get_param( "search" );
        $parent = $request->get_param( "parent" );
        $hide_empty = $request->get_param( "hide_empty" );
        $include = $request->get_param( "include" );

        // Validate taxonomy exists
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return new WP_Error(
                "invalid_taxonomy",
                __( "Taxonomy does not exist", "wvc-theme" ),
                array( "status" => 400 )
            );
        }

        $args = array(
            "taxonomy"   => $taxonomy,
            "number"     => $per_page,
            "offset"     => ( $page - 1 ) * $per_page,
            "hide_empty" => $hide_empty,
        );

        // Support include parameter (single ID, comma-separated list, or array)
        $include_ids = array();
        if ( ! empty( $include ) ) {
            if ( is_array( $include ) ) {
                $include_ids = array_map( "absint", $include );
            } elseif ( is_string( $include ) ) {
                $include_ids = array_filter(
                    array_map(
                        "absint",
                        explode( ",", $include )
                    )
                );
            } else {
                $include_ids = array( absint( $include ) );
            }

            if ( ! empty( $include_ids ) ) {
                $args["include"] = $include_ids;
                // Preserve requested order when possible
                $args["orderby"] = "include";
            }
        }

        if ( ! empty( $search ) ) {
            $args["search"] = $search;
        }

        if ( $parent !== null ) {
            $args["parent"] = $parent;
        }

        $default_term_id = get_option( 'default_' . $taxonomy );

        // Only auto-exclude the default term when no explicit include list is provided.
        if ( $default_term_id && empty( $include_ids ) ) {
            $args["exclude"] = $default_term_id;
        }

        $terms = get_terms( $args );

        if ( is_wp_error( $terms ) ) {
            return $terms;
        }

        $term_manager = WVC_Term_Manager::get_instance();
        $formatted_terms = array();

        foreach ( $terms as $term ) {
            $term_data = $term_manager->get_term( $term->term_id, $taxonomy );
            
            // Add image_url field if term has a thumbnail
            $thumbnail_id = get_term_meta( $term->term_id, "thumbnail_id", true );
            if ( $thumbnail_id ) {
                $image_url = wp_get_attachment_url( $thumbnail_id );
                $term_data["image_url"] = $image_url ? $image_url : "";
            } else {
                $term_data["image_url"] = "";
            }
            
            $formatted_terms[] = $term_data;
        }

        // Get total count for pagination
        $total_args = $args;
        unset( $total_args["number"], $total_args["offset"] );
        $total_terms = get_terms( $total_args );
        $total = is_wp_error( $total_terms ) ? 0 : count( $total_terms );

        $response = rest_ensure_response( $formatted_terms );
        $response->header( "X-WP-Total", $total );
        $response->header( "X-WP-TotalPages", ceil( $total / $per_page ) );

        return $response;
    }

    /**
     * Get arguments for creating a term
     *
     * @return array
     */
    private function get_create_term_args() {
        return array(
            "taxonomy" => array(
                "required"          => true,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "name" => array(
                "required"          => true,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "slug" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_title",
            ),
            "description" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_textarea_field",
            ),
            "uid" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "parent" => array(
                "required"          => false,
                "type"              => "integer",
                "sanitize_callback" => "absint",
                "default"           => 0,
            ),
            "meta" => array(
                "required"          => false,
                "type"              => "object",
                "validate_callback" => function( $param, $request, $key ) {
                    return is_array( $param );
                },
            ),
        );
    }

    /**
     * Get arguments for updating a term
     *
     * @return array
     */
    private function get_update_term_args() {
        return array(
            "taxonomy" => array(
                "required"          => true,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "term_id"  => array(
                "required"          => true,
                "type"              => "integer",
                "sanitize_callback" => "absint",
            ),
            "name" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "slug" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_title",
            ),
            "description" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_textarea_field",
            ),
            "uid" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "parent" => array(
                "required"          => false,
                "type"              => "integer",
                "sanitize_callback" => "absint",
            ),
            "meta" => array(
                "required"          => false,
                "type"              => "object",
                "validate_callback" => function( $param, $request, $key ) {
                    return is_array( $param );
                },
            ),
        );
    }

    /**
     * Get arguments for listing terms
     *
     * @return array
     */
    private function get_list_terms_args() {
        return array(
            "taxonomy" => array(
                "required"          => true,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "per_page" => array(
                "required"          => false,
                "type"              => "integer",
                "sanitize_callback" => "absint",
                "default"           => 10,
                "minimum"           => 1,
                "maximum"           => 100,
            ),
            "page" => array(
                "required"          => false,
                "type"              => "integer",
                "sanitize_callback" => "absint",
                "default"           => 1,
                "minimum"           => 1,
            ),
            "search" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
            ),
            "parent" => array(
                "required"          => false,
                "type"              => "integer",
                "sanitize_callback" => "absint",
            ),
            "hide_empty" => array(
                "required"          => false,
                "type"              => "boolean",
                "sanitize_callback" => "rest_sanitize_boolean",
                "default"           => false,
            ),
            "include" => array(
                "required"          => false,
                "type"              => "string",
                "sanitize_callback" => "sanitize_text_field",
                "description"       => __( "Limit result set to specific term IDs. Accepts a single ID or comma-separated list.", "wvc-theme" ),
            ),
        );
    }

    /**
     * Check if user can create terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function create_term_permission_check( $request ) {
        return $this->permissions_check( $request, "manage_categories" );
    }

    /**
     * Check if user can get terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function get_term_permission_check( $request ) {
        return true; // Public read access
    }

    /**
     * Check if user can update terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function update_term_permission_check( $request ) {
        return $this->permissions_check( $request, "manage_categories" );
    }

    /**
     * Check if user can bulk create/update terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function bulk_upsert_terms_permission_check( $request ) {
        return $this->permissions_check( $request, "manage_categories" );
    }

    /**
     * Check if user can delete terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function delete_term_permission_check( $request ) {
        return $this->permissions_check( $request, "manage_categories" );
    }

    /**
     * Check if user can list terms
     *
     * @param WP_REST_Request $request Full details about the request
     * @return bool|WP_Error
     */
    public function list_terms_permission_check( $request ) {
        return true; // Public read access
    }
}

// Initialize the REST API
WVC_Term_REST_API::get_instance();
